package com.voxeet.sdk.core.network.websocket;

import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.authent.token.TokenResponseProvider;
import com.voxeet.sdk.core.VoxeetHttp;
import com.voxeet.sdk.core.services.authenticate.WebSocketState;
import com.voxeet.sdk.events.error.HttpException;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;

public class WebSocketProxy implements Runnable {

    private final static Handler HANDLER = new Handler();
    private static final int MAXIMUM_RETRY = 60;
    private static final long SOCKET_RETRY_MAX = 2000; //10s
    private static final long SOCKET_RETRY_AFTER = 500; //0.5s
    private static final String TAG = WebSocketProxy.class.getSimpleName();
    private String mVersionName;

    private WebSocketState mState;
    private ExecutorService mExecutorService;
    private String mSocketUrl;
    private int mCount = 0;
    private boolean shouldRetry = true;
    private WebSocketListener mAdapter;
    private List<SocketListener> mListeners;
    private long mSocketCurrentRetryDelay;
    private WebSocket mWebSocket;
    private ConnectListener mConnectListener;
    private boolean isCanceled = false;
    private VoxeetHttp mProvider;
    //private boolean mHasBeenConnectedOrSentInitialConnectFail;

    public WebSocketProxy(@NonNull String socketUrl, String versionName) {
        mVersionName = versionName;
        mState = WebSocketState.CLOSED;
        mSocketUrl = socketUrl;

        mExecutorService = Executors.newSingleThreadExecutor();

        mListeners = new ArrayList<>();
        mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
        mCount = 0;

        mAdapter = new WebSocketListener() {

            @Override
            public void onOpen(@NonNull WebSocket websocket, @NonNull Response response) {
                Log.d(TAG, "onOpen: websocket:=" + websocket + " response:=" + response);
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                mWebSocket = websocket;

                mState = WebSocketState.CONNECTED;

                //reset values to let the app reconnect on error
                mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
                mCount = 0;
                shouldRetry = true;

                try {
                    attemptConnectListenerConnected(websocket);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                onStateChanged(websocket, mState);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onConnect(websocket);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onMessage(@NonNull WebSocket websocket, @NonNull String message) {
                Log.d(TAG, "onMessage: websocket:=" + websocket + " message:=" + message);
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onTextMessage(message);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onFailure(@NonNull WebSocket websocket, @NonNull Throwable cause, Response response) {
                Log.d(TAG, "onFailure: websocket:=" + websocket + " cause:=" + cause + " isCanceled" + isCanceled);
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                cause.printStackTrace();

                Log.d(TAG, "onClosed: cause:=" + cause);
                sendInitialFailed(response);
                checkRetry(websocket, cause.getMessage());
            }

            @Override
            public void onClosing(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                mState = WebSocketState.CLOSING;
                //WebSocketProxy.this.onClosing();
            }

            @Override
            public void onClosed(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                Log.d(TAG, "onClosed: websocket:=" + webSocket + " code:=" + code + " reason" + reason);

                if (isCanceled) {
                    checkCancel();
                    return;
                }

                Log.d(TAG, "onClosed: code:=" + code + " reason:=" + reason);
                //sendInitialFailed(null);
                checkRetry(webSocket, reason);
            }

            private void sendInitialFailed(@Nullable Response response) {
                //mHasBeenConnectedOrSentInitialConnectFail = true;
                HttpException error = HttpException.throwResponse(response);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onError(error);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            private void checkRetry(@NonNull WebSocket websocket, @Nullable String reason) {
                Log.d(TAG, "onClosed: reason := " + reason + " shouldRetry:=" + shouldRetry);

                if (shouldRetry) {
                    if (mCount < MAXIMUM_RETRY) {

                        mState = WebSocketState.CONNECTING;
                        mCount++;
                        mSocketCurrentRetryDelay = Math.min(mSocketCurrentRetryDelay * 2, SOCKET_RETRY_MAX);
                        HANDLER.postDelayed(WebSocketProxy.this, mSocketCurrentRetryDelay);
                    } else {
                        mState = WebSocketState.CLOSED;
                        attemptConnectListenerDisconnected();
                        WebSocketProxy.this.onDisconnected();
                    }
                } else {
                    mState = WebSocketState.CLOSED;
                    onStateChanged(websocket, mState);
                    WebSocketProxy.this.onDisconnected();
                }
            }


            void onStateChanged(WebSocket websocket, WebSocketState newState) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onStateChanged(newState);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        };
    }

    private void attemptConnectListenerDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnectError(new HttpException((Response) null));
            mConnectListener = null;
        }
    }

    private void attemptConnectListenerConnected(WebSocket webSocket) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnect(webSocket);
            mConnectListener = null;
        }
    }

    @Nullable
    private Request setWebSocketRequest() {
        if (isCanceled) {
            checkCancel();
            return null;
        }

        String socketUrl = mSocketUrl;

        String jwtToken = mProvider.getJwtToken();
        if (null != jwtToken) {
            if (!socketUrl.contains("?")) socketUrl += "?Token=" + jwtToken;
            else socketUrl += "Token=" + jwtToken;

            Log.d(TAG, "setWebSocketRequest: socketUrl := " + socketUrl);

            return TokenResponseProvider.setUserAgent(new Request.Builder(), mVersionName)
                    .url(socketUrl)
                    .build();
        }
        return null;
    }

    public void connect(@NonNull VoxeetHttp provider, @Nullable ConnectListener connectListener) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        mConnectListener = connectListener;
        mProvider = provider;

        Request request = setWebSocketRequest();

        if (null != request) {
            mState = WebSocketState.CONNECTING;

            mWebSocket = provider.getClient().newWebSocket(request, mAdapter);
        }
    }

    private void onDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        for (SocketListener listener : mListeners) {
            listener.onDisconnected();
        }
    }

    public void addListener(@NonNull SocketListener listener) {
        if (!mListeners.contains(listener)) {
            mListeners.add(listener);
        }
    }

    public void removeListener(@NonNull SocketListener listener) {
        mListeners.remove(listener);
    }

    @Override
    public void run() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        connect(mProvider, mConnectListener);
    }

    public boolean isOpen() {
        return WebSocketState.CONNECTED.equals(mState);//mWebSocket != null && mWebSocket.isOpen();
    }

    public boolean sendPing() {
        Log.e(TAG, "sendPing:");
        return false;//mWebSocket != null && mWebSocket.request().sendPing().getState() == WebSocketState.OPEN;
    }

    public void disconnect() {
        mState = WebSocketState.CLOSED;
        //prevent reconnection
        mCount = MAXIMUM_RETRY;
        shouldRetry = false;

        //disconnect
        if (null != mWebSocket) {
            mWebSocket.close(1000, "Closing socket from client");
            //mWebSocket.disconnect(WebSocketCloseCode.NORMAL, null, CLOSE_VOXEET_AFTER_MILLISECS);
        }

        for (SocketListener listener : mListeners) {
            listener.onClose();
        }

        mExecutorService.shutdown();
    }

    public void removeListeners() {
        mListeners.clear();
        mAdapter = null;
    }

    public WebSocketState getState() {
        return mState;
    }

    public WebSocket getWebSocket() {
        return mWebSocket;
    }

    private void checkCancel() {
        Log.d(TAG, "checkCancel: isCanceled:=" + isCanceled + " isOpen:=" + isOpen());
        if (isCanceled && isOpen()) {
            disconnect();
        }
    }

    public void cancel() {
        removeListeners();
        isCanceled = true;
    }
}
