/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.components.colorpicker;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;

import com.vaadin.shared.ui.colorpicker.Color;
import com.vaadin.ui.Component;
import com.vaadin.ui.CustomField;

/**
 * A component that represents color selection history within a color picker.
 *
 * @since 7.0.0
 */
public class ColorPickerHistory extends CustomField<Color> {

    private static final String STYLENAME = "v-colorpicker-history";

    private static final int ROWS = 4;

    private static final int COLUMNS = 15;

    /** Temporary color history for when the component is detached. */
    private final ArrayBlockingQueue<Color> tempHistory = new ArrayBlockingQueue<>(
            ROWS * COLUMNS);

    @Override
    protected Component initContent() {
        setPrimaryStyleName(STYLENAME);

        ColorPickerGrid grid = new ColorPickerGrid(ROWS, COLUMNS);
        grid.setWidth("100%");
        grid.setPosition(0, 0);
        grid.addValueChangeListener(event -> {
            setValue(grid.getValue());
            fireEvent(new ValueChangeEvent<Color>(ColorPickerHistory.this,
                    event.getOldValue(), event.isUserOriginated()));
        });

        return grid;
    }

    @Override
    protected ColorPickerGrid getContent() {
        return (ColorPickerGrid) super.getContent();
    }

    @Override
    public void attach() {
        super.attach();
        createColorHistoryIfNecessary();
    }

    @Override
    public void detach() {
        // refresh the tempHistory to match the one saved to session
        ArrayBlockingQueue<Color> colorHistory = getColorHistory();
        if (!tempHistory.equals(colorHistory)) {
            tempHistory.clear();
            for (Color color : colorHistory) {
                tempHistory.offer(color);
            }
        }
        super.detach();
    }

    private void createColorHistoryIfNecessary() {
        List<Color> tempColors = new ArrayList<>(tempHistory);
        if (getSession().getAttribute("colorPickerHistory") == null) {
            getSession().setAttribute("colorPickerHistory",
                    new ArrayBlockingQueue<Color>(ROWS * COLUMNS));
        }
        for (Color color : tempColors) {
            setValue(color);
        }
        tempHistory.clear();
    }

    @SuppressWarnings("unchecked")
    private ArrayBlockingQueue<Color> getColorHistory() {
        if (isAttached()) {
            Object colorHistory = getSession()
                    .getAttribute("colorPickerHistory");
            if (colorHistory instanceof ArrayBlockingQueue<?>) {
                return (ArrayBlockingQueue<Color>) colorHistory;
            }
        }
        return tempHistory;
    }

    @Override
    public void setHeight(String height) {
        super.setHeight(height);
        getContent().setHeight(height);
    }

    @Override
    public Color getValue() {
        ArrayBlockingQueue<Color> colorHistory = getColorHistory();
        if (colorHistory.isEmpty()) {
            return null;
        }
        List<Color> colorList = new ArrayList<>(colorHistory);

        // return the newest one
        return colorList.get(colorList.size() - 1);
    }

    @Override
    protected void doSetValue(Color color) {

        ArrayBlockingQueue<Color> colorHistory = getColorHistory();

        // remove and add to the top
        colorHistory.remove(color);
        if (!colorHistory.offer(color)) {
            // history is full, remove one and try again
            colorHistory.poll();
            colorHistory.offer(color);
        }

        List<Color> colorList = new ArrayList<>(colorHistory);

        // Invert order of colors
        Collections.reverse(colorList);

        // Create 2d color map
        Color[][] colors = new Color[ROWS][COLUMNS];
        Iterator<Color> iter = colorList.iterator();

        for (int row = 0; row < ROWS; row++) {
            for (int col = 0; col < COLUMNS; col++) {
                if (iter.hasNext()) {
                    colors[row][col] = iter.next();
                } else {
                    colors[row][col] = Color.WHITE;
                }
            }
        }

        getContent().setColorGrid(colors);
        getContent().markAsDirty();
    }

    /**
     * Gets the history.
     *
     * @return the history
     */
    public List<Color> getHistory() {
        ArrayBlockingQueue<Color> colorHistory = getColorHistory();
        Color[] array = colorHistory.toArray(new Color[colorHistory.size()]);
        return Collections.unmodifiableList(Arrays.asList(array));
    }

    /**
     * Checks if the history contains given color.
     *
     * @param c
     *            the color
     *
     * @return true, if successful
     */
    public boolean hasColor(Color c) {
        return getColorHistory().contains(c);
    }
}
