/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui;

import com.vaadin.data.HasValue.ValueChangeEvent;
import com.vaadin.shared.ui.ValueChangeMode;

/**
 * Implemented by components which support value change modes.
 *
 * @since 8.0
 */
public interface HasValueChangeMode extends Component {
    /**
     * Sets the mode how the TextField triggers {@link ValueChangeEvent}s.
     *
     * @param valueChangeMode
     *            the new mode
     *
     * @see ValueChangeMode
     */
    public void setValueChangeMode(ValueChangeMode valueChangeMode);

    /**
     * Returns the currently set {@link ValueChangeMode}.
     *
     * @return the mode used to trigger {@link ValueChangeEvent}s.
     *
     * @see ValueChangeMode
     */
    public ValueChangeMode getValueChangeMode();

    /**
     * Sets how often {@link ValueChangeEvent}s are triggered when the
     * {@link ValueChangeMode} is set to either {@link ValueChangeMode#LAZY} or
     * {@link ValueChangeMode#TIMEOUT}.
     *
     * @param valueChangeTimeout
     *            timeout in milliseconds, must be greater or equal to 0
     * @throws IllegalArgumentException
     *             if given timeout is smaller than 0
     *
     * @see ValueChangeMode
     */
    public void setValueChangeTimeout(int valueChangeTimeout);

    /**
     * Returns the currently set timeout, in milliseconds, for how often
     * {@link ValueChangeEvent}s are triggered if the current
     * {@link ValueChangeMode} is set to either {@link ValueChangeMode#LAZY} or
     * {@link ValueChangeMode#TIMEOUT}.
     *
     * @return the timeout in milliseconds of how often
     *         {@link ValueChangeEvent}s are triggered.
     */
    public int getValueChangeTimeout();

}
