/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.event;

import java.lang.reflect.Method;

import com.vaadin.ui.Component;
import com.vaadin.util.ReflectTools;

/**
 * An event that is fired when an item is expanded in a listing component that
 * displays hierarchical data.
 *
 * @author Vaadin Ltd
 * @since 8.1
 * @param <T>
 *            the expanded item's type
 */
public class ExpandEvent<T> extends Component.Event
        implements HasUserOriginated {

    private final T expandedItem;

    private final boolean userOriginated;

    /**
     * Construct an expand event.
     *
     * @param source
     *            the hierarchical component this event originated from
     * @param expandedItem
     *            the item that was expanded
     * @param userOriginated
     *            whether the expand was triggered by a user interaction or the
     *            server
     */
    public ExpandEvent(Component source, T expandedItem,
            boolean userOriginated) {
        super(source);
        this.expandedItem = expandedItem;
        this.userOriginated = userOriginated;
    }

    /**
     * Get the expanded item that triggered this event.
     *
     * @return the expanded item
     */
    public T getExpandedItem() {
        return expandedItem;
    }

    @Override
    public boolean isUserOriginated() {
        return userOriginated;
    }

    /**
     * Item expand event listener.
     *
     * @param <T>
     *            the expanded item's type
     * @since 8.1
     */
    @FunctionalInterface
    public interface ExpandListener<T> extends SerializableEventListener {

        public static final Method EXPAND_METHOD = ReflectTools.findMethod(
                ExpandListener.class, "itemExpand", ExpandEvent.class);

        /**
         * Callback method for when an item has been expanded.
         *
         * @param event
         *            the expand event
         */
        public void itemExpand(ExpandEvent<T> event);
    }
}
