package com.vaadin.copilot;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ai.AICommandHandler;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.analytics.AnalyticsInterceptor;
import com.vaadin.copilot.feedback.FeedbackHandler;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeHandler;
import com.vaadin.copilot.ide.IdePluginCommandHandler;
import com.vaadin.copilot.ide.OpenComponentInIDE;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.plugins.accessibilitychecker.AccessibilityCheckerMessageHandler;
import com.vaadin.copilot.plugins.devsetup.DevSetupHandler;
import com.vaadin.copilot.plugins.docs.DocsHandler;
import com.vaadin.copilot.plugins.i18n.I18nHandler;
import com.vaadin.copilot.plugins.info.InfoHandler;
import com.vaadin.copilot.plugins.themeeditor.ThemeEditorMessageHandler;
import com.vaadin.flow.di.Lookup;
import com.vaadin.flow.internal.DevModeHandlerManager;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import elemental.json.Json;
import elemental.json.JsonObject;

/**
 * The main code for Copilot for a given VaadinSession instance.
 *
 * <p>
 * One instance of this class is created for each VaadinSession which mean it is
 * ok to use and store VaadinSession specific data in this class and command
 * classes it uses, and in project manager etc.
 */
public class CopilotSession {

    ProjectManager projectManager;

    private List<CopilotCommand> commands;

    private final CopilotIDEPlugin idePlugin;

    private SourceSyncChecker sourceSyncChecker;

    /**
     * Create a new CopilotSession for the given VaadinSession.
     *
     * @param vaadinSession
     *            the VaadinSession
     * @param devToolsInterface
     *            used to send messages back to the browser
     * @throws IOException
     *             if an error occurs
     */
    public CopilotSession(VaadinSession vaadinSession, DevToolsInterface devToolsInterface) throws IOException {
        VaadinServletContext context = Copilot.getContext(vaadinSession);

        ApplicationConfiguration applicationConfiguration = ApplicationConfiguration.get(context);
        CopilotIDEPlugin.setFolderInLaunchedModule(applicationConfiguration.getProjectFolder().toPath());
        CopilotIDEPlugin.setDevToolsInterface(devToolsInterface);
        if (ProjectFileManager.getInstance() == null
                || ProjectFileManager.getInstance().getApplicationConfiguration() != applicationConfiguration) {
            ProjectFileManager.initialize(applicationConfiguration);
        }
        projectManager = new ProjectManager(applicationConfiguration, vaadinSession);
        idePlugin = CopilotIDEPlugin.getInstance();

        Optional<DevModeHandlerManager> devModeHandlerManagerOptional = getDevModeHandlerManager(context);
        if (!applicationConfiguration.isProductionMode() && sourceSyncChecker == null
                && devModeHandlerManagerOptional.isPresent()) {
            sourceSyncChecker = new SourceSyncChecker(projectManager, devModeHandlerManagerOptional.get());
        }

        setupCommands(applicationConfiguration, context);
    }

    public void handleConnect(DevToolsInterface devToolsInterface) {
        devToolsInterface.send(Copilot.PREFIX + "init", null);
        for (CopilotCommand copilotCommand : commands) {
            copilotCommand.handleConnect(devToolsInterface);
        }
    }

    /**
     * Handle a message from the client.
     *
     * @param command
     *            the command
     * @param data
     *            the data, specific to the command
     * @param devToolsInterface
     *            used to send messages back to the browser
     */
    public void handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command == null) {
            throw new NullPointerException("command is null");
        }

        boolean canBeParallel = false;
        canBeParallel = commands.stream().anyMatch(c -> c.canBeParallelCommand(command));
        if (canBeParallel) {
            handleMessageAsync(command, data, devToolsInterface);
        } else {
            handleMessageSync(command, data, devToolsInterface);
        }
    }

    private void handleMessageAsync(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        CompletableFuture<Boolean>[] futures = commands.stream()
                .map(copilotCommand -> CompletableFuture
                        .runAsync(() -> copilotCommand.handleMessage(command, data, devToolsInterface)))
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).thenRunAsync(() -> {
            boolean handled = Stream.of(futures).anyMatch(CompletableFuture::join);
            if (!handled) {
                JsonObject respData = Json.createObject();
                if (data.hasKey(CopilotCommand.KEY_REQ_ID)) {
                    respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));
                }
                data.put("error", "Unknown command " + command);
                devToolsInterface.send("unknown-command", data);
            }
        });
    }

    private void handleMessageSync(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        for (CopilotCommand copilotCommand : commands) {
            if (copilotCommand.handleMessage(command, data, devToolsInterface)) {
                return;
            }
        }
        JsonObject respData = Json.createObject();
        if (data.hasKey(CopilotCommand.KEY_REQ_ID)) {
            respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));
        }
        data.put("error", "Unknown command " + command);
        devToolsInterface.send("unknown-command", data);
    }

    private void setupCommands(ApplicationConfiguration applicationConfiguration, VaadinServletContext context) {
        commands = List.of(
                // This must be first as it is more of an interceptor than a
                // handler
                new AnalyticsInterceptor(), new IdeHandler(idePlugin),
                new ErrorHandler(AnalyticsClient.getInstance().isEnabled()), new OpenComponentInIDE(projectManager),
                new ProjectFileHandler(projectManager), new AICommandHandler(projectManager),
                new UserInfoHandler(context), new I18nHandler(projectManager),
                new IdePluginCommandHandler(projectManager), new ApplicationInitializer(projectManager, context),
                new MachineConfigurationHandler(), new ThemeEditorMessageHandler(projectManager),
                new RouteHandler(projectManager, context), new UiServiceHandler(projectManager, context),
                new AccessibilityCheckerMessageHandler(projectManager), new InfoHandler(applicationConfiguration),
                new FeedbackHandler(), new JavaRewriteHandler(projectManager, sourceSyncChecker), new PreviewHandler(),
                new JavaParserHandler(), new DocsHandler(), new DevSetupHandler(idePlugin),
                new HotswapDownloadHandler(projectManager));
    }

    private Optional<DevModeHandlerManager> getDevModeHandlerManager(VaadinContext context) {
        return Optional.ofNullable(context).map(ctx -> ctx.getAttribute(Lookup.class))
                .map(lu -> lu.lookup(DevModeHandlerManager.class));
    }
}
