package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_AD_OBJECT
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_OBJECT_NOT_FOUND
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import gateway.v1.adResponse
import headerbidding.v1.HeaderBiddingAdMarkupOuterClass.*
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext

internal class AndroidLoad(
    private val defaultDispatcher: CoroutineDispatcher,
    private val getAdRequest: GetAdRequest,
    private val getAdPlayerConfigRequest: GetAdPlayerConfigRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val handleGatewayAdResponse: HandleGatewayAdResponse,
    private val sessionRepository: SessionRepository,
    private val gatewayClient: GatewayClient,
    private val adRepository: AdRepository,
) : Load {
    override suspend fun invoke(
        context: Context,
        placement: String,
        opportunityId: ByteString,
        headerBiddingAdMarkup: HeaderBiddingAdMarkup
    ): LoadResult = withContext(defaultDispatcher) {
        if (!sessionRepository.isSdkInitialized) {
            return@withContext LoadResult.Failure(error = UnityAdsLoadError.INITIALIZE_FAILED, reason = REASON_NOT_INITIALIZED)
        }

        val response = if (headerBiddingAdMarkup.adData.isEmpty) {
            // waterfall
            sessionRepository.incrementLoadRequestCount()
            val loadAdRequest = getAdRequest(placement, opportunityId)
            val requestPolicy = getRequestPolicy()
            val gatewayAdResponse = gatewayClient.request(request = loadAdRequest, requestPolicy = requestPolicy, operationType = OperationType.LOAD)
            gatewayAdResponse.payload.adResponse
        } else {
            // header bidding
            sessionRepository.incrementLoadRequestAdmCount()
            val adPlayerConfigRequest = getAdPlayerConfigRequest(placement, opportunityId, headerBiddingAdMarkup.configurationToken)
            val requestPolicy = getRequestPolicy()
            val gatewayAdPlayerConfigResponse = gatewayClient.request(request = adPlayerConfigRequest, requestPolicy = requestPolicy, operationType = OperationType.LOAD_HEADER_BIDDING)
            if (gatewayAdPlayerConfigResponse.hasError()) {
                return@withContext LoadResult.Failure(
                    error = UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_INTERNAL_ERROR,
                    reason = REASON_GATEWAY,
                    reasonDebug = gatewayAdPlayerConfigResponse.error.errorText)
            }
            val response = gatewayAdPlayerConfigResponse.payload.adPlayerConfigResponse
            adResponse {
                adData = headerBiddingAdMarkup.adData
                adDataVersion = headerBiddingAdMarkup.adDataVersion
                trackingToken = response.trackingToken
                impressionConfiguration = response.impressionConfiguration
                impressionConfigurationVersion = response.impressionConfigurationVersion
                webviewConfiguration = response.webviewConfiguration
                adDataRefreshToken = response.adDataRefreshToken
                if (response.hasError()) {
                    error = response.error
                }
            }
        }

        val adLoadResponse = handleGatewayAdResponse(opportunityId, response, context, placement)
        when (adLoadResponse) {
            is LoadResult.Success -> {
                val adObject = adRepository.getAd(opportunityId)
                if (adObject == null) {
                    LoadResult.Failure(error = UnityAdsLoadError.INTERNAL_ERROR, message = MSG_AD_OBJECT, reason = REASON_AD_OBJECT_NOT_FOUND)
                } else {
                    LoadResult.Success(adObject = adObject)
                }
            }
            is LoadResult.Failure -> adLoadResponse
        }
    }
}