package com.unity3d.ads.adplayer

import android.util.Base64
import com.unity3d.ads.adplayer.model.*
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.adplayer.model.ShowStatus
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.ExecuteAdViewerRequest
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.BRIDGE_SEND_EVENT_FAILED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_CODE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_VIEWER
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.services.core.device.Storage
import com.unity3d.services.core.device.StorageEventInfo
import com.unity3d.services.core.network.mapper.toResponseHeadersMap
import com.unity3d.services.core.network.model.RequestType
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import org.json.JSONObject

private val SHOW_EVENTS = arrayOf(
    ExposedFunctionLocation.STARTED,
    ExposedFunctionLocation.CLICKED,
    ExposedFunctionLocation.COMPLETED,
    ExposedFunctionLocation.FAILED,
    ExposedFunctionLocation.CANCEL_SHOW_TIMEOUT
)

private val LOAD_EVENTS = arrayOf(
    ExposedFunctionLocation.LOAD_COMPLETE,
    ExposedFunctionLocation.LOAD_ERROR,
)

private val REQUEST_EVENTS = arrayOf(
    ExposedFunctionLocation.REQUEST_GET,
    ExposedFunctionLocation.REQUEST_POST,
    ExposedFunctionLocation.REQUEST_HEAD
)

/**
 * Class responsible for handling ad playback in a WebView.
 *
 * @property bridge The WebViewBridge used for communication with the WebView.
 * @property deviceInfoRepository The repository used for retrieving device information.
 * @property sessionRepository The repository used for managing session data.
 * @property dispatcher The CoroutineDispatcher used for running coroutines.
 */
internal class WebViewAdPlayer(
    private val bridge: WebViewBridge,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val sessionRepository: SessionRepository,
    private val executeAdViewerRequest: ExecuteAdViewerRequest,
    private val dispatcher: CoroutineDispatcher,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    adPlayerScope: CoroutineScope,
) : AdPlayer {
    private val storageEventCallback: (StorageEventInfo) -> Unit = {
        scope.launch {
            bridge.sendEvent(OnStorageEvent(
                eventType = it.eventType,
                storageType = it.storageType,
                value = it.value
            ))
        }
    }

    private val scopeCancellationHandler = CoroutineExceptionHandler { _, _ ->
        Storage.removeStorageEventCallback(storageEventCallback)
    }

    override val scope = adPlayerScope + dispatcher + CoroutineName("WebViewAdPlayer") + scopeCancellationHandler

    override val onShowEvent = bridge.onInvocation
        .filter { it.location in SHOW_EVENTS }
        .map {
            val event = when (it.location) {
                ExposedFunctionLocation.STARTED -> ShowEvent.Started
                ExposedFunctionLocation.CLICKED -> ShowEvent.Clicked
                ExposedFunctionLocation.COMPLETED -> ShowEvent.Completed(
                    when (it.parameters.first() as? String) {
                        "COMPLETED" -> ShowStatus.COMPLETED
                        "SKIPPED" -> ShowStatus.SKIPPED
                        else -> ShowStatus.ERROR
                    }
                )
                ExposedFunctionLocation.FAILED -> {
                    val jsonObject = it.parameters.first() as JSONObject
                    val errorCode = jsonObject.optInt("code")
                    val errorMessage = jsonObject.optString("message")
                    ShowEvent.Error(errorMessage, errorCode, REASON_AD_VIEWER)
                }
                ExposedFunctionLocation.CANCEL_SHOW_TIMEOUT -> ShowEvent.CancelTimeout
                else -> throw IllegalStateException("Unexpected location: ${it.location}")
            }
            it.handle()
            event
        }

    override val onLoadEvent = bridge.onInvocation
        .filter { it.location in LOAD_EVENTS }
        .map {
            it.handle()

            if (it.location == ExposedFunctionLocation.LOAD_ERROR) {
                val jsonObject = it.parameters.first() as JSONObject
                val errorCode = jsonObject.optInt("code")
                val errorMessage = jsonObject.optString("message")
                LoadEvent.Error(errorMessage, errorCode)
            } else {
                LoadEvent.Completed
            }
        }
        .shareIn(scope, SharingStarted.Eagerly, 1)
        .take(1)

    override val updateCampaignState: Flow<Pair<ByteArray, Int>> = bridge.onInvocation
        .filter { it.location == ExposedFunctionLocation.UPDATE_CAMPAIGN_STATE }
        .map {
            it.handle()
            val campaignJSON = it.parameters.first() as JSONObject
            val data = campaignJSON.optString("data")
            val dataByteString = data.toByteArray(Charsets.ISO_8859_1)
            val dataVersion = campaignJSON.optInt("dataVersion")
            dataByteString to dataVersion
        }

    private val onBroadcastEvents = bridge.onInvocation
        .filter { it.location == ExposedFunctionLocation.BROADCAST_EVENT }
        .map {
            it.handle()
            it.parameters.first().toString()
        }

    val onRequestEvents = bridge.onInvocation
        .filter { it.location in REQUEST_EVENTS }
        .map {
            it.handle()

            val id = it.parameters.first() as String
            val url = it.parameters.getOrNull(1) as String?

            val type = when (it.location) {
                ExposedFunctionLocation.REQUEST_GET -> RequestType.GET
                ExposedFunctionLocation.REQUEST_POST -> RequestType.POST
                ExposedFunctionLocation.REQUEST_HEAD -> RequestType.HEAD
                else -> throw IllegalStateException("Unexpected location: ${it.location}")
            }

            try {
                val response = executeAdViewerRequest(type, it.parameters)
                val body = when (val responseBody = response.body) {
                    is String -> responseBody
                    is ByteArray -> String(responseBody, Charsets.UTF_8)
                    else -> null
                }
                val bridgeResponse = listOf(
                    id,
                    response.urlString,
                    body,
                    response.statusCode,
                    response.headers.toResponseHeadersMap()
                )
                bridge.sendEvent(OnWebRequestComplete(bridgeResponse))
            } catch (e: Exception) {
                val bridgeResponse = listOf(id, url, e.message ?: "")
                bridge.sendEvent(OnWebRequestFailed(bridgeResponse))
            }
        }

    init {
        Storage.addStorageEventCallback(storageEventCallback)

        // Listen to calls of `broadcastEvent`, and emit them to the broadcast event channel.
        onBroadcastEvents.onEach(AdPlayer.broadcastEventChannel::emit).launchIn(scope)

        // Listen to calls of `request`, and emit them to the request event channel.
        onRequestEvents.launchIn(scope)

        // Listen to the broadcast event channel, and send the events to the webview.
        AdPlayer.broadcastEventChannel.onEach(::onBroadcastEvent).launchIn(scope)
    }

    // region AdViewer API

    override suspend fun requestShow() {
        val dynamicDeviceInfo = deviceInfoRepository.dynamicDeviceInfo

        val showOptions = JSONObject().also {
            it.put("orientation", deviceInfoRepository.orientation)
            it.put("connectionType", deviceInfoRepository.connectionTypeStr)
            it.put("isMuted", deviceInfoRepository.ringerMode != 2)
            it.put("volume", dynamicDeviceInfo.android.volume)
            it.put("privacy", sessionRepository.getPrivacy().toBase64())
            it.put("privacyFsm", sessionRepository.getPrivacyFsm().toBase64())
            // TODO: Discuss with AdViewer and iOS how we should really pass around allowedPii structure.
            it.put("allowedPii", deviceInfoRepository.allowedPii.value.toByteString().toBase64())
        }

        bridge.request("webview", "show", showOptions)
    }

    // endregion AdViewer API

    // region Events

    private suspend fun sendEvent(getEvent: () -> WebViewEvent) {
        val loadEvent = onLoadEvent.single()
        if (loadEvent is LoadEvent.Error) {
            sendDiagnosticEvent(
                event = BRIDGE_SEND_EVENT_FAILED,
                tags = mapOf(
                    REASON to REASON_AD_VIEWER,
                    REASON_DEBUG to loadEvent.message,
                    REASON_CODE to loadEvent.errorCode.toString(),
                )
            )
            return
        }

        val event = getEvent()
        bridge.sendEvent(event)
    }

    override suspend fun sendMuteChange(isMuted: Boolean) = sendEvent {
        OnMuteChangeEvent(isMuted)
    }

    override suspend fun sendVisibilityChange(isVisible: Boolean) = sendEvent {
        OnVisibilityChangeEvent(isVisible)
    }

    override suspend fun sendVolumeChange(volume: Double) = sendEvent {
        OnVolumeChangeEvent(volume)
    }

    override suspend fun sendUserConsentChange(value: ByteArray) = sendEvent {
        OnUserConsentChangeEvent(Base64.encodeToString(value, Base64.NO_WRAP))
    }

    override suspend fun sendPrivacyFsmChange(value: ByteArray) = sendEvent {
        OnPrivacyFsmChangeEvent(Base64.encodeToString(value, Base64.NO_WRAP))
    }

    override suspend fun onBroadcastEvent(event: String) = sendEvent {
        val json = JSONObject(event)
        val eventType = json.getString("eventType")
        val data = json.optString("data")

        OnBroadcastEvent(eventType, data)
    }

    override suspend fun onAllowedPiiChange(value: ByteArray) = sendEvent {
        OnAllowedPiiChangeEvent(Base64.encodeToString(value, Base64.NO_WRAP))
    }

    // endregion Events
}
