package com.unity3d.ads.adplayer

import android.content.Intent
import com.unity3d.ads.core.data.datasource.VolumeSettingsChange
import com.unity3d.ads.core.data.model.SessionChange
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.core.data.repository.SessionRepository
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*

/**
 * An internal class representing an Android fullscreen webview ad player.
 *
 * This class implements the [AdPlayer] via [WebViewAdPlayer] and [FullscreenAdPlayer] interfaces. It provides methods to show
 * the ad, destroy the ad player, and send various events related to the ad player.
 *
 * @property webViewAdPlayer The underlying webview ad player.
 * @property opportunityId The ID of the ad opportunity.
 * @property webViewContainer The Android webview container.
 * @property deviceInfoRepository The repository for device information.
 * @property sessionRepository The repository for session information.
 */
internal class AndroidFullscreenWebViewAdPlayer(
    private val webViewAdPlayer: WebViewAdPlayer,
    private val opportunityId: String,
    private val webViewContainer: AndroidWebViewContainer,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val sessionRepository: SessionRepository,
) : AdPlayer by webViewAdPlayer, FullscreenAdPlayer {
    /**
     * Show method is responsible for showing the [FullScreenWebViewDisplay], which is an activity. It will contain the webview (using [AndroidWebViewContainer])
     * and the ad will be played in the webview via the ad viewer.
     *
     * @param showOptions The show options for displaying the activity. They are not the one passed to the ad viewer.
     * @throws IllegalArgumentException if the showOptions is not an instance of AndroidShowOptions. (KMM future compatibility)
     */
    override fun show(showOptions: ShowOptions) {
        require(showOptions is AndroidShowOptions)
        val context = showOptions.context

        val intent = Intent(context, FullScreenWebViewDisplay::class.java).also {
            it.putExtra("opportunityId", opportunityId)
            it.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION or Intent.FLAG_ACTIVITY_NEW_TASK)
        }

        // Captures messages from the display (activity on Android) and routes them to the webview ad player.
        displayMessages
            .filter { it.opportunityId == opportunityId }
            .onEach(::displayEventsRouter)
            .launchIn(scope)

        deviceInfoRepository.volumeSettingsChange
            .onEach(::handleVolumeSettingsChange)
            .launchIn(scope)

        webViewAdPlayer.onShowEvent
            .filter { it is ShowEvent.Completed || it is ShowEvent.Error }
            .onEach { destroy() }
            .launchIn(scope)

        sessionRepository.onChange
            .onEach(::handleSessionChange)
            .launchIn(scope)

        context.startActivity(intent)
    }

    private suspend fun handleVolumeSettingsChange(change: VolumeSettingsChange) {
        when (change) {
            is VolumeSettingsChange.MuteChange -> webViewAdPlayer.sendMuteChange(change.isMuted)
            is VolumeSettingsChange.VolumeChange -> webViewAdPlayer.sendVolumeChange(change.volume)
        }
    }

    private suspend fun handleSessionChange(change: SessionChange) {
        when (change) {
            is SessionChange.UserConsentChange -> webViewAdPlayer.sendUserConsentChange(change.value.toByteArray())
            is SessionChange.PrivacyFsmChange -> webViewAdPlayer.sendPrivacyFsmChange(change.value.toByteArray())
        }
    }

    override suspend fun destroy() {
        // Notify the display (activity on Android) to finish (destroy itself).
        displayMessages.emit(DisplayMessage.DisplayFinishRequest(opportunityId))

        // Destroy the webview container.
        webViewContainer.destroy()

        super<AdPlayer>.destroy()
    }

    private fun displayEventsRouter(displayMessage: DisplayMessage) = scope.launch {
        when (displayMessage) {
            is DisplayMessage.DisplayReady -> webViewAdPlayer.requestShow()
            is DisplayMessage.WebViewInstanceRequest -> displayMessages.emit(DisplayMessage.WebViewInstanceResponse(displayMessage.opportunityId, webViewContainer.webView))
            is DisplayMessage.VisibilityChanged -> webViewAdPlayer.sendVisibilityChange(displayMessage.isVisible)
            // todo: report error message?
            is DisplayMessage.DisplayError -> destroy()
            else -> Unit
        }
    }

    companion object {
        /**
         * A flow of display messages that is used to communicate with displays. It is currently used by Android with [FullScreenWebViewDisplay].
         */
        val displayMessages = MutableSharedFlow<DisplayMessage>()
    }
}