package com.unity3d.ads.adplayer

import androidx.annotation.CallSuper
import com.unity3d.ads.adplayer.model.LoadEvent
import com.unity3d.ads.core.data.model.ShowEvent
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow

/**
 * This interface represents an ad player that can be used to play advertisements.
 * It provides methods to control the player's behavior and events for tracking ad playback.
 **/
interface AdPlayer {
    /**
     * Represents a CoroutineScope object associated to the lifetime of the AdPlayer. It can be used to launch Flows
     * that will be cancelled when the AdPlayer is destroyed.
     **/
    val scope: CoroutineScope

    /**
     * Represents a Flow of ShowEvent objects that can be used to track ad playback.
     **/
    val onShowEvent: Flow<ShowEvent>

    /**
     * Represents the event that is triggered when the AdViewer html has finished loading.
     *
     * Only triggered once per ad view.
     *
     * If an error occurs, it will throw an exception.
     **/
    val onLoadEvent: Flow<LoadEvent>
    val updateCampaignState: Flow<Pair<ByteArray, Int>>

    // region AdViewer API

    /**
     * Sends a request to the AdViewer to start playing the ad.
     **/
    suspend fun requestShow()

    fun show(showOptions: ShowOptions) {
        throw NotImplementedError()
    }

    @CallSuper
    suspend fun destroy() {
        // Cancel the scope. Any coroutines launched in the scope will be cancelled (such as flow listeners).
        scope.cancel()
    }

    // endregion AdViewer API

    // region Events

    suspend fun sendMuteChange(isMuted: Boolean)
    suspend fun sendVisibilityChange(isVisible: Boolean)
    suspend fun sendVolumeChange(volume: Double)
    suspend fun sendUserConsentChange(value: ByteArray)
    suspend fun sendPrivacyFsmChange(value: ByteArray)
    suspend fun onAllowedPiiChange(value: ByteArray)
    suspend fun onBroadcastEvent(event: String)

    companion object {
        val broadcastEventChannel = MutableSharedFlow<String>()
    }

    // endregion Events
}