package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.unity3d.ads.IUnityAdsLoadListener
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_OPPORTUNITY_ID
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_OPPORTUNITY_ID_USED
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_TIMEOUT
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.extensions.duration
import com.unity3d.ads.core.extensions.getInitializationStateString
import com.unity3d.ads.core.extensions.toDiagnosticReason
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.properties.SdkProperties
import com.unity3d.services.core.request.metrics.AdOperationMetric
import gateway.v1.AdResponseOuterClass
import gateway.v1.adResponse
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import java.util.UUID

internal class LegacyLoadUseCase(
    private val dispatcher: CoroutineDispatcher,
    private val load: Load,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sessionRepository: SessionRepository,
    private val adRepository: AdRepository,
) {
    suspend operator fun invoke(
        context: Context,
        placement: String,
        loadOptions: UnityAdsLoadOptions,
        unityLoadListener: IUnityAdsLoadListener?
    ) {
        val startTime = loadStart()
        DeviceLog.debug("Unity Ads Load Start for placement $placement")

        val opportunityId = getOpportunityId(loadOptions)
        if (opportunityId == null) {
            loadFailure(startTime, UnityAdsLoadError.INVALID_ARGUMENT, MSG_OPPORTUNITY_ID, placement, unityLoadListener)
            return
        }

        val opportunityIdByteString = UUID.fromString(opportunityId).toByteString()
        if (adRepository.hasOpportunityId(opportunityIdByteString)) {
            loadFailure(startTime, UnityAdsLoadError.INVALID_ARGUMENT, MSG_OPPORTUNITY_ID_USED, placement, unityLoadListener)
            return
        }

        // For header bidding, the ad markup is passed in the load options and contains the adData
        val adMarkup = getAdMarkup(loadOptions)
        val adResponse = if (adMarkup.isNullOrBlank()) {
            AdResponseOuterClass.AdResponse.getDefaultInstance()
        } else {
            adResponse {
                adData = ByteString.copyFromUtf8(adMarkup)
            }
        }

        val useTimeout = true // sessionRepository.nativeConfiguration.featureFlags.loadTimeoutEnabled
        val timeoutMillis = sessionRepository.nativeConfiguration.adOperations.loadTimeoutMs.toLong()

        try {
            val loadResult = if (useTimeout) {
                withTimeoutOrNull(timeoutMillis) {
                    load(context, placement, opportunityIdByteString, adResponse)
                } ?: LoadResult.Failure(UnityAdsLoadError.TIMEOUT, MSG_TIMEOUT + placement)
            } else {
                load(context, placement, opportunityIdByteString, adResponse)
            }

            when(loadResult) {
                is LoadResult.Success -> loadSuccess(startTime, placement, unityLoadListener)
                is LoadResult.Failure -> loadFailure(startTime, loadResult.error, loadResult.message ?: "", placement, unityLoadListener)
            }
        } catch (e: Exception) {
            loadFailure(startTime, UnityAdsLoadError.INTERNAL_ERROR, e.message ?: "", placement, unityLoadListener)
        }

    }

    private fun getOpportunityId(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_OBJECT_ID)?.toString()
    }

    private fun getAdMarkup(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_AD_MARKUP)?.toString()
    }

    private suspend fun loadStart(): Long {
        val startTime = System.nanoTime()
        sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_STARTED)
        return startTime
    }

    private suspend fun loadSuccess(
        startTime: Long,
        placement: String,
        unityLoadListener: IUnityAdsLoadListener?
    ) {
        DeviceLog.debug("Unity Ads Load Success for placement: $placement")
        sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_SUCCESS, value = startTime.duration(), tags = getTags())
        withContext(dispatcher) {
            unityLoadListener?.onUnityAdsAdLoaded(placement)
        }
    }

    private suspend fun loadFailure(
        startTime: Long,
        reason: UnityAdsLoadError,
        message: String = "",
        placement: String,
        unityLoadListener: IUnityAdsLoadListener?
    ) {
        DeviceLog.debug("Unity Ads Load Failure for placement: $placement reason: $reason :: $message")
        val event = if (reason == UnityAdsLoadError.TIMEOUT) SendDiagnosticEvent.LOAD_TIMEOUT else SendDiagnosticEvent.LOAD_FAILURE
        sendDiagnosticEvent(event = event, value = startTime.duration() ,tags = getTags(reason))
        withContext(dispatcher) {
            unityLoadListener?.onUnityAdsFailedToLoad(placement, reason, message)
        }
    }

    private fun getTags(reason: UnityAdsLoadError? = null): Map<String, String> {
        val initState = SdkProperties.getCurrentInitializationState()
        val tags = mutableMapOf(AdOperationMetric.INIT_STATE to initState.getInitializationStateString().toString())
        if (reason != null) {
            tags[REASON] = reason.toDiagnosticReason()
            tags[OPERATION] = OperationType.LOAD.toString()
        }
        return tags
    }

    companion object {
        const val KEY_OBJECT_ID = "objectId"
        const val KEY_AD_MARKUP = "adMarkup"
    }
}