package com.unity3d.ads.core.domain

import android.content.Context
import android.util.Base64
import com.google.protobuf.ByteString
import com.unity3d.ads.UnityAds
import com.unity3d.ads.adplayer.AdPlayerError
import com.unity3d.ads.adplayer.AndroidFullscreenWebViewAdPlayer
import com.unity3d.ads.adplayer.WebViewAdPlayer
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.CampaignState
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_COMMUNICATION_FAILURE
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_COMMUNICATION_FAILURE_WITH_DETAILS
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_NO_FILL
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.CampaignStateRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.extensions.toISO8859String
import com.unity3d.services.UnityAdsConstants
import gateway.v1.AdResponseOuterClass.AdResponse
import gateway.v1.OperativeEventRequestOuterClass
import gateway.v1.operativeEventErrorData
import gateway.v1.timestamps
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext
import java.net.URI
import java.net.URL

internal class HandleGatewayAndroidAdResponse(
    private val adRepository: AdRepository,
    private val getWebViewContainerUseCase: AndroidGetWebViewContainerUseCase,
    private val getWebViewBridge: GetWebViewBridgeUseCase,
    private val defaultDispatcher: CoroutineDispatcher,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val getHandleAndroidInvocationsUseCase: HandleAndroidInvocationsUseCase,
    private val sessionRepository: SessionRepository,
    private val campaignStateRepository: CampaignStateRepository,
    private val executeAdViewerRequest: ExecuteAdViewerRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getOperativeEventApi: GetOperativeEventApi
) : HandleGatewayAdResponse {
    override suspend fun invoke(opportunityId: ByteString, response: AdResponse, context: Context, placementId: String): LoadResult {

        var adPlayer: AndroidFullscreenWebViewAdPlayer? = null

        try {
            if (response.hasError()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE
                )
            }

            if (response.adData.isEmpty) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.NO_FILL,
                    message = MSG_NO_FILL
                )
            }

            if (!response.hasWebviewConfiguration()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE
                )
            }
//        require(response.webviewConfiguration.entryPoint.isEmpty()) { "No entryPoint" }
            // fixme: It can be empty, which means we should use last known entry point

            val url = URI(response.webviewConfiguration.entryPoint)
            val webViewDefaultQuery = url.query
            val webViewUrl = URL(
                url.scheme,
                url.host,
                url.path
            ).toString() + UnityAdsConstants.DefaultUrls.AD_PLAYER_QUERY_PARAMS + webViewDefaultQuery
            val base64ImpressionConfiguration = Base64.encodeToString(
                response.impressionConfiguration.toByteArray(),
                Base64.NO_WRAP
            )
            val webviewContainer = getWebViewContainerUseCase()
            val webviewBridge = getWebViewBridge(webviewContainer)
            val webViewAdPlayer = WebViewAdPlayer(
                webviewBridge,
                deviceInfoRepository,
                sessionRepository,
                executeAdViewerRequest,
                defaultDispatcher,
                sendDiagnosticEvent
            )

             adPlayer = AndroidFullscreenWebViewAdPlayer(
                webViewAdPlayer = webViewAdPlayer,
                webViewContainer = webviewContainer,
                opportunityId = opportunityId.toISO8859String(),
                deviceInfoRepository = deviceInfoRepository,
                sessionRepository = sessionRepository
            )

            deviceInfoRepository.allowedPii
                .onEach(webViewAdPlayer::onAllowedPiiChange)
                .launchIn(webViewAdPlayer.scope)

            webViewAdPlayer.updateCampaignState.onEach { (data, dataVersion) ->
                val state = campaignStateRepository.getState(opportunityId)
                    ?.copy(data = data, dataVersion = dataVersion)
                    ?: CampaignState(data, dataVersion, placementId, timestamps {}, timestamps {})

                campaignStateRepository.updateState(opportunityId, state)
            }.launchIn(webViewAdPlayer.scope)

            val adObject = AdObject(
                opportunityId = opportunityId,
                placementId = placementId,
                trackingToken = response.trackingToken,
                fullscreenAdPlayer = adPlayer
                )

            getHandleAndroidInvocationsUseCase(
                onInvocations = webviewBridge.onInvocation,
                adData = response.adData.toBase64(),
                impressionConfig = base64ImpressionConfiguration,
                adDataRefreshToken = response.adDataRefreshToken.toBase64(),
                adObject = adObject,
                onSubscription = { webviewContainer.loadUrl(webViewUrl) }
            ).launchIn(webViewAdPlayer.scope)

            try {
                webViewAdPlayer.loadEvent.await()
                campaignStateRepository.setLoadTimestamp(opportunityId)
            } catch (t: Throwable) {
                val loadFailure = LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    throwable = t
                )
                if (t is CancellationException) {
                    if (t.cause is AdPlayerError) {
                        return loadFailure.copy(
                            message = MSG_COMMUNICATION_FAILURE_WITH_DETAILS.format(t.cause?.message)
                        )
                    } else {
                        throw t
                    }
                }
                withContext(NonCancellable) {
                    cleanup(t, opportunityId, response, adPlayer)
                }
                return loadFailure
            }

            adRepository.addAd(opportunityId, adObject)

            return LoadResult.Success(adObject)
        } catch (t: CancellationException) {
            withContext(NonCancellable) {
                cleanup(t, opportunityId, response, adPlayer)
            }
            throw t
        }
    }

    private suspend fun cleanup(
        t: Throwable,
        opportunityId: ByteString,
        response: AdResponse,
        adPlayer: AndroidFullscreenWebViewAdPlayer?
    ) {
        val operativeEventErrorData = operativeEventErrorData {
            errorType = OperativeEventRequestOuterClass.OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_UNSPECIFIED
            message = t.message ?: ""
        }
        getOperativeEventApi(
            operativeEventType = OperativeEventRequestOuterClass.OperativeEventType.OPERATIVE_EVENT_TYPE_LOAD_ERROR,
            opportunityId = opportunityId,
            trackingToken = response.trackingToken,
            additionalEventData = operativeEventErrorData.toByteString()
        )
        adPlayer?.destroy()
    }
}