package com.unity3d.ads.core.domain

import android.util.Base64
import com.google.protobuf.ByteString
import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.adplayer.ExposedFunctionLocation
import com.unity3d.ads.adplayer.Invocation
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.repository.CampaignStateRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.utils.ContinuationFromCallback
import com.unity3d.services.core.api.Storage
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventType
import gateway.v1.PrivacyUpdateRequestOuterClass.PrivacyUpdateRequest
import gateway.v1.copy
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.flow.update
import org.json.JSONObject
import kotlin.coroutines.suspendCoroutine

internal class HandleAndroidInvocationsUseCase(
    private val getAndroidAdPlayerContext: GetAndroidAdPlayerContext,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val refresh: Refresh,
    private val handleOpenUrl: HandleOpenUrl,
    private val sessionRepository: SessionRepository,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val campaignStateRepository: CampaignStateRepository,
    private val sendPrivacyUpdateRequest: SendPrivacyUpdateRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
) {
    operator fun invoke(
        onInvocations: SharedFlow<Invocation>,
        adData: String,
        adDataRefreshToken: String,
        impressionConfig: String,
        adObject: AdObject,
        onSubscription: suspend () -> Unit,
    ) = onInvocations.onSubscription { onSubscription() }.onEach {
        when (it.location) {
            ExposedFunctionLocation.GET_AD_CONTEXT -> it.handle {
                buildMap {
                    put(KEY_AD_DATA, adData)
                    put(KEY_IMPRESSION_CONFIG, impressionConfig)
                    put(KEY_AD_DATA_REFRESH_TOKEN, adDataRefreshToken)
                    put(KEY_NATIVE_CONTEXT, getAndroidAdPlayerContext())
                }
            }

            ExposedFunctionLocation.GET_SCREEN_HEIGHT -> it.handle { deviceInfoRepository.staticDeviceInfo().screenHeight }
            ExposedFunctionLocation.GET_SCREEN_WIDTH -> it.handle { deviceInfoRepository.staticDeviceInfo().screenWidth }
            ExposedFunctionLocation.GET_CONNECTION_TYPE -> it.handle { deviceInfoRepository.connectionTypeStr }
            ExposedFunctionLocation.GET_DEVICE_VOLUME -> it.handle { deviceInfoRepository.dynamicDeviceInfo.android.volume }
            ExposedFunctionLocation.GET_DEVICE_MAX_VOLUME -> it.handle { deviceInfoRepository.dynamicDeviceInfo.android.maxVolume }
            ExposedFunctionLocation.SEND_OPERATIVE_EVENT -> it.handle {
                getOperativeEventApi(
                    adObject = adObject,
                    operativeEventType = OperativeEventType.OPERATIVE_EVENT_TYPE_SPECIFIED_BY_AD_PLAYER,
                    additionalEventData = Base64.decode(it.parameters[0] as String, Base64.NO_WRAP).toByteString()
                )
            }

            ExposedFunctionLocation.OPEN_URL -> it.handle {
                val url = it.parameters[0] as String
                val params = it.parameters[1] as JSONObject
                val packageName = params.optString(KEY_PACKAGE_NAME)

                handleOpenUrl(url, packageName)
            }

            ExposedFunctionLocation.STORAGE_WRITE -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.write(
                        it.parameters[0] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            }

            ExposedFunctionLocation.STORAGE_CLEAR -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.clear(it.parameters[0] as String, ContinuationFromCallback(continuation))
                }
            }

            ExposedFunctionLocation.STORAGE_DELETE -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.delete(
                        it.parameters[0] as String,
                        it.parameters[1] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            }

            ExposedFunctionLocation.STORAGE_READ -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.read(it.parameters[0] as String, ContinuationFromCallback(continuation))
                }
            }

            ExposedFunctionLocation.STORAGE_GET_KEYS -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.getKeys(
                        it.parameters[0] as String,
                        it.parameters[1] as String,
                        it.parameters[2] as Boolean,
                        ContinuationFromCallback(continuation)
                    )
                }
            }

            ExposedFunctionLocation.STORAGE_GET -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.get(
                        it.parameters[0] as String,
                        it.parameters[1] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            }

            ExposedFunctionLocation.STORAGE_SET -> it.handle {
                suspendCoroutine { continuation ->
                    Storage.set(
                        it.parameters[0] as String,
                        it.parameters[1] as String,
                        it.parameters[2],
                        ContinuationFromCallback(continuation)
                    )
                }
            }

            ExposedFunctionLocation.GET_PRIVACY_FSM -> it.handle {
                sessionRepository.getPrivacyFsm().toBase64()
            }

            ExposedFunctionLocation.SET_PRIVACY_FSM -> it.handle {
                sessionRepository.setPrivacyFsm(Base64.decode(it.parameters[0] as String, Base64.NO_WRAP).toByteString())
            }

            ExposedFunctionLocation.SET_PRIVACY -> it.handle {
                sessionRepository.setPrivacy(Base64.decode(it.parameters[0] as String, Base64.NO_WRAP).toByteString())
            }

            ExposedFunctionLocation.GET_PRIVACY -> it.handle {
                sessionRepository.getPrivacy().toBase64()
            }

            ExposedFunctionLocation.GET_ALLOWED_PII -> it.handle {
                Base64.encodeToString(deviceInfoRepository.allowedPii.value.toByteArray(), Base64.NO_WRAP)
            }

            ExposedFunctionLocation.SET_ALLOWED_PII -> it.handle {
                val allowedPiiUpdated = it.parameters[0] as Map<String, Boolean>
                deviceInfoRepository.allowedPii.update { allowedPii ->
                    allowedPii.copy {
                        allowedPiiUpdated["idfa"]?.let(::idfa::set)
                        allowedPiiUpdated["idfv"]?.let(::idfv::set)
                    }
                }
            }

            ExposedFunctionLocation.GET_SESSION_TOKEN -> it.handle {
                sessionRepository.sessionToken.toBase64()
            }

            ExposedFunctionLocation.MARK_CAMPAIGN_STATE_SHOWN -> it.handle {
                campaignStateRepository.setShowTimestamp(adObject.opportunityId)
            }

            ExposedFunctionLocation.REFRESH_AD_DATA -> it.handle {

                val refreshTokenByteString = if (it.parameters.isEmpty()) {
                    ByteString.EMPTY
                } else {
                    val refreshTokenJson = it.parameters[0] as JSONObject
                    val refreshToken = refreshTokenJson.optString(KEY_AD_DATA_REFRESH_TOKEN)
                    refreshToken.fromBase64()
                }

                val adRefreshResponse = refresh(refreshTokenByteString, adObject.opportunityId)

                if (adRefreshResponse.hasError()) throw IllegalArgumentException("Refresh failed")

                buildMap {
                    put(KEY_AD_DATA, adRefreshResponse.adData.toBase64())
                    put(KEY_AD_DATA_REFRESH_TOKEN, adRefreshResponse.adDataRefreshToken.toBase64())
                    put(KEY_TRACKING_TOKEN, adRefreshResponse.trackingToken.toBase64())
                }
            }

            ExposedFunctionLocation.UPDATE_TRACKING_TOKEN -> it.handle {
                val updateTrackingToken = it.parameters[0] as JSONObject
                val token: String? = updateTrackingToken.optString("trackingToken")

                if (!token.isNullOrEmpty()) {
                    adObject.trackingToken = token.fromBase64()
                }
            }

            ExposedFunctionLocation.SEND_PRIVACY_UPDATE_REQUEST -> it.handle {
                val base64Proto = it.parameters[0] as String
                val privacyUpdateRequest = PrivacyUpdateRequest.parseFrom(Base64.decode(base64Proto, Base64.NO_WRAP))

                val response = sendPrivacyUpdateRequest(privacyUpdateRequest)

                Base64.encodeToString(response.toByteArray(), Base64.NO_WRAP)
            }

            ExposedFunctionLocation.SEND_DIAGNOSTIC_EVENT -> it.handle {
                val event = it.parameters[0] as String
                val tags = it.parameters[1] as JSONObject
                val tagsMap = buildMap {
                    tags.keys().forEach { key ->
                        put(key, tags.getString(key))
                    }
                }

                sendDiagnosticEvent(event = event, tags = tagsMap)
            }

            else -> {}
        }
    }

    companion object {
        const val KEY_AD_DATA = "adData"
        const val KEY_AD_DATA_REFRESH_TOKEN = "adDataRefreshToken"
        const val KEY_TRACKING_TOKEN = "trackingToken"
        const val KEY_IMPRESSION_CONFIG = "impressionConfig"
        const val KEY_NATIVE_CONTEXT = "nativeContext"

        const val KEY_PACKAGE_NAME = "packageName"
    }
}