package com.unity3d.ads.core.data.repository

import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.log.DeviceLog
import gateway.v1.DiagnosticEventRequestOuterClass.DiagnosticEventType
import gateway.v1.DiagnosticEventRequestOuterClass.DiagnosticEvent
import gateway.v1.NativeConfigurationOuterClass.DiagnosticEventsConfiguration
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.update
import java.util.Timer
import kotlin.concurrent.timerTask

class AndroidDiagnosticEventRepository : DiagnosticEventRepository {

    private val batch = MutableStateFlow<MutableList<DiagnosticEvent>>(value = mutableListOf())
    private val flushTimer: Timer = Timer()
    private var maxBatchSize = Int.MAX_VALUE
    private val allowedEvents = mutableSetOf<DiagnosticEventType>()
    private val blockedEvents = mutableSetOf<DiagnosticEventType>()
    private val enabled = MutableStateFlow(false)
    private val configured = MutableStateFlow(false)

    private val _diagnosticEvents = MutableSharedFlow<List<DiagnosticEvent>>(
        replay = UnityAdsConstants.SharedFlow.REPLAY,
        extraBufferCapacity = UnityAdsConstants.SharedFlow.EXTRA_CAPACITY,
        onBufferOverflow = BufferOverflow.DROP_OLDEST
    )

    override val diagnosticEvents = _diagnosticEvents.asSharedFlow()

    override fun addDiagnosticEvent(diagnosticEvent: DiagnosticEvent) {
        if (!configured.value) {
            batch.value += diagnosticEvent
        } else {
            if (!enabled.value) return
            batch.value += diagnosticEvent
            if (batch.value.size >= maxBatchSize) {
                flush()
            }
        }
    }

    override fun flush() {
        val events = batch.value
        DeviceLog.debug("Unity Ads Sending diagnostic batch enabled: ${enabled.value} size: ${events.size} :: $events")
        events.asSequence()
            .filter { allowedEvents.isEmpty() || it.eventType in allowedEvents }
            .filter { it.eventType !in blockedEvents }
            .toList()

        clear()

        if (events.isNotEmpty()) {
            _diagnosticEvents.tryEmit(events)
        }
    }

    override fun clear() {
        batch.update { mutableListOf() }
    }

    override fun configure(diagnosticsEventsConfiguration: DiagnosticEventsConfiguration) {
       enabled.value = diagnosticsEventsConfiguration.enabled
        if (!enabled.value) {
            clear()
            return
        }
        this.maxBatchSize = diagnosticsEventsConfiguration.maxBatchSize
        this.allowedEvents.addAll(diagnosticsEventsConfiguration.allowedEventsList)
        this.blockedEvents.addAll(diagnosticsEventsConfiguration.blockedEventsList)

        val maxBatchIntervalMs = diagnosticsEventsConfiguration.maxBatchIntervalMs

        flushTimer.scheduleAtFixedRate(
            timerTask()
            { flush() },
            maxBatchIntervalMs.toLong(),
            maxBatchIntervalMs.toLong()
        )

        flush()
        configured.value = true
    }
}