package com.unity3d.ads.core.data.datasource

import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.content.Context
import android.content.pm.ApplicationInfo
import android.content.pm.PackageManager
import android.hardware.Sensor
import android.hardware.SensorManager
import android.media.MediaCodecInfo
import android.media.MediaCodecList
import android.os.Build
import android.os.ext.SdkExtensions
import android.webkit.WebSettings
import com.google.protobuf.ByteString
import com.unity3d.ads.BuildConfig
import com.unity3d.ads.core.data.datasource.AndroidDynamicDeviceInfoDataSource.Companion.DIRECTORY_MEM_INFO
import com.unity3d.ads.core.data.datasource.AndroidDynamicDeviceInfoDataSource.Companion.DIRECTORY_MODE_READ
import com.unity3d.ads.core.data.model.StorageType
import com.unity3d.services.core.device.AdvertisingId
import com.unity3d.services.core.device.Device.MemoryInfoType
import com.unity3d.services.core.device.MimeTypes
import com.unity3d.services.core.device.OpenAdvertisingId
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.misc.Utilities
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.properties.SdkProperties
import gateway.v1.StaticDeviceInfoKt.android
import gateway.v1.StaticDeviceInfoOuterClass
import gateway.v1.staticDeviceInfo
import java.io.*
import java.security.MessageDigest
import java.security.cert.CertificateException
import java.security.cert.CertificateFactory
import java.security.cert.X509Certificate
import java.util.*
import java.util.regex.Pattern
import javax.security.auth.x500.X500Principal
import kotlin.math.roundToInt


/**
 * Class responsible for retrieving all device data that isn't expected to change during a session.
 *
 * This data is not expected to ever change during a session.
 * A single fetch should be sufficient and the value be safely reused and up to date.
 */
@Suppress("DEPRECATION")
class AndroidStaticDeviceInfoDataSource(
    val context: Context,
    private val idfiStore: ByteStringDataSource,
    private val auidStore: ByteStringDataSource,
    private val glInfoStore: ByteStringDataSource,
    private val analyticsDataSource: AnalyticsDataSource
) : StaticDeviceInfoDataSource {

    private val DEBUG_CERT = X500Principal("CN=Android Debug,O=Android,C=US")

    /**
     * Fill out all the fields required for the [StaticDeviceInfoOuterClass.StaticDeviceInfo] protobuf
     */
    override suspend fun fetch(): StaticDeviceInfoOuterClass.StaticDeviceInfo = staticDeviceInfo {
        bundleId = getAppName()
        bundleVersion = getAppVersion()
        appDebuggable = isAppDebuggable()
        rooted = isRooted()
        osVersion = getOsVersion()
        deviceMake = getManufacturer()
        deviceModel = getModel()
        webviewUa = getWebViewUserAgent()
        screenDensity = getScreenDensity()
        screenWidth = getScreenWidth()
        screenHeight = getScreenHeight()
        screenSize = getScreenLayout()
        stores += getStores()
        totalDiskSpace = getTotalSpace(getFileForStorageType(StorageType.EXTERNAL))
        totalRamMemory = getTotalMemory()
        cpuModel = getCPUModel()
        cpuCount = getCPUCount()
        gpuModel = getGPUModel()
        android = fetchAndroidStaticDeviceInfo()
    }

    private fun fetchAndroidStaticDeviceInfo(): StaticDeviceInfoOuterClass.StaticDeviceInfo.Android = android {
        apiLevel = getApiLevel()
        versionCode = getVersionCode()
        androidFingerprint = getFingerprint()
        appInstaller = getInstallerPackageName()
        apkDeveloperSigningCertificateHash = getCertificateFingerprint()
        buildBoard = getBoard()
        buildBrand = getBrand()
        buildDevice = getDevice()
        buildDisplay = getDisplay()
        buildFingerprint = getFingerprint()
        buildHardware = getHardware()
        buildHost = getHost()
        buildBootloader = getBootloader()
        buildProduct = getProduct()
        extensionVersion = getExtensionVersion()
        getBuildId()?.let { buildId = it }
    }

    /**
     * https://developer.android.com/reference/android/os/Build.VERSION#SDK_INT
     *
     * This value never changes while a device is booted, but it may increase when the hardware manufacturer provides an OTA update.
     *
     * @return the device SDK version of the software currently running on this hardware device
     */
    private fun getApiLevel(): Int {
        return Build.VERSION.SDK_INT
    }

    /**
     * https://developer.android.com/reference/android/os/Build.VERSION#RELEASE
     *
     * E.g., "1.0" or "3.4b5" or "bananas".
     * This field is an opaque string. Do not assume that its value has any particular structure or that values of RELEASE from different releases can be somehow ordered.
     *
     * @return the user-visible version string.
     */
    override fun getOsVersion(): String {
        return Build.VERSION.RELEASE ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#MANUFACTURER
     *
     * @return the manufacturer of the product/hardware.
     */
    override fun getManufacturer(): String {
        return Build.MANUFACTURER ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#MODEL
     *
     * @return the end-user-visible name for the end product.
     */
    override fun getModel(): String {
        return Build.MODEL ?: ""
    }

    private fun getScreenLayout(): Int {
        return context.resources.configuration.screenLayout
    }

    /**
     * https://developers.google.com/android/reference/com/google/android/gms/ads/identifier/AdvertisingIdClient.Info#getId()
     *
     * @return the id from Googles AdvertisingIdClient
     */
    fun getAdvertisingTrackingId(): String {
        return AdvertisingId.getAdvertisingTrackingId() ?: ""
    }

    /**
     * The Open Anonymous Device Identifier (OAID) is a user-resettable unique identifier for Android devices. It was introduced by the Mobile Security Alliance (MSA), China Information and Communication Research Institute, and device manufacturers, as a privacy-preserving alternative to non-resettable device identifiers like IMEI.
     * The OAID is usually used for ad measurement and install attribution on devices where Google Play Services aren't available (meaning where Google Advertising ID does not exist).
     *
     * @return OpenAdvertisingId or an empty string if it does not exist
     */
    private fun getOpenAdvertisingTrackingId(): String {
        return OpenAdvertisingId.getOpenAdvertisingTrackingId() ?: ""
    }

    /**
     *
     * @return if open ad tracking is enabled on the device
     * @see getOpenAdvertisingTrackingId()
     */
    fun isLimitOpenAdTrackingEnabled(): Boolean {
        return OpenAdvertisingId.getLimitedOpenAdTracking()
    }

    /**
     * This is considered a PII field
     *
     * @return the UUID created at install time
     */
    override suspend fun getIdfi(): ByteString {
        return idfiStore.get().data
    }

    override val analyticsUserId: String?
        get() = analyticsDataSource.userId

    /**
     * This is considered a PII field
     *
     * @return the AUID provided by LevelPlay
     */
    override suspend fun getAuid(): ByteString {
        return auidStore.get().data
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#density
     *
     * The logical density of the display. This is a scaling factor for the Density Independent Pixel unit, where one DIP is one pixel on an approximately 160 dpi screen (for example a 240x320, 1.5"x2" screen), providing the baseline of the system's display. Thus on a 160dpi screen this density value will be 1; on a 120 dpi screen it would be .75; etc.
     *
     * This value does not exactly follow the real screen size (as given by xdpi and ydpi), but rather is used to scale the size of the overall UI in steps based on gross changes in the display dpi. For example, a 240x320 screen will have a density of 1 even if its width is 1.8", 1.3", etc. However, if the screen resolution is increased to 320x480 but the screen size remained 1.5"x2" then the density would be increased (probably to 1.5).
     *
     * @return the screen density from [android.util.DisplayMetrics.density] or 0 if unavailable
     */
    private fun getDisplayMetricDensity(): Float {
        return context.resources?.displayMetrics?.density ?: (0).toFloat()
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#densityDpi
     *
     * The screen density expressed as dots-per-inch.
     *
     * May be either:
     * DENSITY_LOW = 120
     * DENSITY_MEDIUM = 160
     * DENSITY_HIGH = 240
     *
     * @return the screen density from [android.util.DisplayMetrics.densityDpi] or 0 if unavailable
     */
    private fun getScreenDensity(): Int {
        return context.resources?.displayMetrics?.densityDpi ?: -1
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#widthPixels
     *
     * @return the screen width from [android.util.DisplayMetrics.widthPixels] or 0 if unavailable
     */
    private fun getScreenWidth(): Int {
        return context.resources?.displayMetrics?.widthPixels ?: -1
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#heightPixels
     *
     * @return the screen height from [android.util.DisplayMetrics.heightPixels] or 0 if unavailable
     */
    private fun getScreenHeight(): Int {
        return context.resources?.displayMetrics?.heightPixels ?: -1
    }

    /**
     * Looks for the su binary in PATH
     *
     * @return if the device is rooted
     */
    private fun isRooted(): Boolean {
        return try {
            searchPathForBinary(BINARY_SU)
        } catch (e: Exception) {
            DeviceLog.exception("Rooted check failed", e)
            false
        }
    }

    /**
     * @return if a specific binary is found in PATH
     * @see isRooted
     */
    private fun searchPathForBinary(binary: String): Boolean {
        val paths = System.getenv(ENVIRONMENT_VARIABLE_PATH)?.split(":".toRegex())?.dropLastWhile { it.isEmpty() }
            ?.toTypedArray() ?: return false
        for (path in paths) {
            val pathDir = File(path)
            if (pathDir.exists() && pathDir.isDirectory) {
                val pathDirFiles = pathDir.listFiles()
                if (pathDirFiles != null) {
                    for (fileInPath in pathDirFiles) {
                        if (fileInPath.name == binary) {
                            return true
                        }
                    }
                }
            }
        }
        return false
    }

    /**
     * https://developer.android.com/reference/android/content/pm/PackageManager#GET_SIGNATURES
     *
     * @return information about the signatures included in the package.
     */
    @SuppressLint("PackageManagerGetSignatures")
    @Deprecated("This constant was deprecated in API level 28. Use GET_SIGNING_CERTIFICATES instead")
    private fun getCertificateFingerprint(): String {
        val pm = context.packageManager
        val pkgName = context.packageName
        try {
            val pinfo = pm.getPackageInfo(pkgName, PackageManager.GET_SIGNATURES)
            val signatures = pinfo.signatures
            if (signatures != null && signatures.isNotEmpty()) {
                val cf = CertificateFactory.getInstance("X.509")
                val stream = ByteArrayInputStream(signatures[0].toByteArray())
                val cert = cf.generateCertificate(stream) as X509Certificate
                val messageDigest = MessageDigest.getInstance(ALGORITHM_SHA1)
                val publicKey = messageDigest.digest(cert.encoded)
                return Utilities.toHexString(publicKey)
            }
        } catch (e: java.lang.Exception) {
            DeviceLog.exception("Exception when signing certificate fingerprint", e)
        }
        return ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#BOARD
     *
     * @return the name of the underlying board, like "goldfish".
     */
    fun getBoard(): String {
        return Build.BOARD ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#BOOTLOADER
     *
     * @return the system bootloader version number.
     */
    fun getBootloader(): String {
        return Build.BOOTLOADER ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#BRAND
     *
     * @return the consumer-visible brand with which the product/hardware will be associated, if any.
     */
    fun getBrand(): String {
        return Build.BRAND ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#DISPLAY
     *
     * @return A build ID string meant for displaying to the user
     */
    fun getDisplay(): String {
        return Build.DISPLAY ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#DEVICE
     *
     * @return the name of the industrial design.
     */
    fun getDevice(): String {
        return Build.DEVICE ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#HOST
     *
     * @return the name of the hardware (from the kernel command line or /proc).
     */
    fun getHardware(): String {
        return Build.HARDWARE ?: ""
    }

    /**
     * This build prop is built by build info shell script during the compilation of the ROM
     *
     * @return info about internal engineering build
     */
    fun getHost(): String {
        return Build.HOST ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#PRODUCT
     *
     * @return the name of the overall product.
     */
    fun getProduct(): String {
        return Build.PRODUCT ?: ""
    }

    /**
     * https://developer.android.com/reference/android/os/Build#FINGERPRINT
     *
     * @return a string that uniquely identifies this build. Do not attempt to parse this value.
     */
    private fun getFingerprint(): String {
        return Build.FINGERPRINT ?: ""
    }

    /**
     * https://developer.android.com/reference/android/content/pm/PackageManager#getInstallerPackageName(java.lang.String)
     *
     * Retrieve the package name of the application that installed a package. This identifies which market the package came from.
     */
    @Deprecated("This method was deprecated in API level 30. use getInstallSourceInfo")
    private fun getInstallerPackageName(): String {
        return context.packageManager.getInstallerPackageName(context.packageName) ?: ""
    }

    /**
     * Will return the list of supported ABIS from the correct API depending on OS version
     *
     * @return the list of supported ABIS
     */
    fun getSupportedAbis(): List<String> {
        return if (getApiLevel() < 21) {
            getOldAbiList()
        } else {
            getNewAbiList()
        }
    }

    /**
     * @return the list of available sensors
     */
    fun getSensorList(): List<Sensor> {
        val sensorManager = context
            .getSystemService(Context.SENSOR_SERVICE) as SensorManager
        return sensorManager.getSensorList(Sensor.TYPE_ALL)
    }

    /**
     * The model name of the device's primary system-on-chip.
     */
    private fun getCPUModel(): String {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            Build.SOC_MODEL
        } else {
            File("/proc/cpuinfo").readLines().last()
        }
    }

    /**
     * @return the devices number of available processors
     */
    private fun getCPUCount(): Long {
        return Runtime.getRuntime().availableProcessors().toLong()
    }

    /**
     * The model name of the device's GPU.
     * Source is GL_RENDERER from OpenGL ES
     */
    private suspend fun getGPUModel(): String {
        return glInfoStore.get().data.toString(Charsets.ISO_8859_1)
    }


    /**
     * https://developer.android.com/reference/android/os/Build#ID
     *
     * @return either a changelist number, or a label like "M4-rc20".
     */
    fun getBuildId(): String? {
        return Build.ID
    }

    /**
     * https://developer.android.com/reference/android/os/Build.VERSION#INCREMENTAL
     *
     * @return the internal value used by the underlying source control to represent this build. E.g., a perforce changelist number or a git hash.
     */
    fun getBuildVersionIncremental(): String? {
        return Build.VERSION.INCREMENTAL
    }

    /**
     * https://developer.android.com/reference/android/os/Build#SUPPORTED_ABIS
     *
     * @return the list of supported ABIS
     */
    private fun getOldAbiList(): List<String> {
        val abiList = mutableListOf<String>()
        abiList.add(Build.CPU_ABI)
        abiList.add(Build.CPU_ABI2)
        return abiList
    }

    /**
     * https://developer.android.com/reference/android/os/Build#SUPPORTED_ABIS
     *
     * @return the list of supported ABIS
     */
    @TargetApi(21)
    private fun getNewAbiList(): ArrayList<String> {
        val abiList = ArrayList<String>()
        abiList.addAll(listOf(*Build.SUPPORTED_ABIS))
        return abiList
    }

    /**
     * @return the devices webview user agent
     */
    private fun getWebViewUserAgent(): String = WebSettings.getDefaultUserAgent(context)

    /**
     * @return the SDK version code
     */
    private fun getVersionCode(): Int = BuildConfig.VERSION_CODE

    /**
     * @return the list of app stores the device supports
     */
    private fun getStores(): List<String> = listOf(STORE_GOOGLE)

    /**
     * @return a timestamp of when the app started up
     */
    private fun getAppStartTime(): Long = SdkProperties.getInitializationTimeEpoch()

    /**
     * @return the SDK version name
     */
    private fun getVersionName(): String = BuildConfig.VERSION_NAME

    /**
     * @return if the SDK is in test mode
     */
    private fun isTestMode(): Boolean = SdkProperties.isTestMode()

    /**
     * @return the device platform in this case android
     */
    private fun getPlatform(): String = PLATFORM_ANDROID

    /**
     * @return the GameId
     */
    private fun getGameId(): String = ClientProperties.getGameId() ?: ""

    /**
     * @return the total amount of physical RAM, in kilobytes.
     */
    fun getTotalMemory(): Long {
        return getMemoryInfo(MemoryInfoType.TOTAL_MEMORY)
    }

    /**
     * @param infoType the [MemoryInfoType] for the amount to be returned
     * @return amount of memory depending on [MemoryInfoType] passed in
     */
    private fun getMemoryInfo(infoType: MemoryInfoType): Long {
        val lineNumber: Int = when (infoType) {
            MemoryInfoType.TOTAL_MEMORY -> 1
            MemoryInfoType.FREE_MEMORY -> 2
            else -> -1
        }
        var line: String? = null
        RandomAccessFile(
            DIRECTORY_MEM_INFO,
            DIRECTORY_MODE_READ
        ).use {
            for (i in 0 until lineNumber) {
                line = it.readLine()
            }
        }
        return getMemoryValueFromString(line)
    }

    /**
     * Converts a memory value from a [String] to a [Long]
     *
     * @param memVal the [String] memory value to be converted
     * @return the [Long] value of the inputted memory [String]
     */
    private fun getMemoryValueFromString(memVal: String?): Long {
        if (memVal != null) {
            val p = Pattern.compile("(\\d+)")
            val m = p.matcher(memVal)
            var value: String? = null
            while (m.find()) {
                value = m.group(1)
            }
            return value?.toLong() ?: -1
        }
        return 0
    }

    /**
     * https://developer.android.com/reference/java/io/File#getTotalSpace()
     *
     * @param file the [File] on which to calculate the total space available
     * @return the size of the partition named by this abstract pathname
     */
    fun getTotalSpace(file: File?): Long {
        return if (file != null && file.exists()) {
            (file.totalSpace / 1024).toFloat().roundToInt().toLong()
        } else -1
    }

    /**
     * https://developer.android.com/training/data-storage
     *
     * @param storageType the [StorageType] for which to calculate the available space
     * @return the available space in kilobytes
     */
    private fun getFileForStorageType(storageType: StorageType): File? {
        return when (storageType) {
            StorageType.INTERNAL -> ClientProperties.getApplicationContext().cacheDir
            StorageType.EXTERNAL -> ClientProperties.getApplicationContext().externalCacheDir
            else -> {
                DeviceLog.error("Unhandled storagetype: $storageType")
                null
            }
        }
    }

    /**
     * @return the app packageName
     */
    override fun getAppName(): String {
        return context.packageName
    }

    /**
     * @return the app versionName or [APP_VERSION_FAKE] if null
     */
    private fun getAppVersion(): String {
        val pkgName = context.packageName
        val pm = context.packageManager

        return try {
            if (pm.getPackageInfo(pkgName, 0).versionName == null) {
                APP_VERSION_FAKE
            } else {
                pm.getPackageInfo(pkgName, 0).versionName
            }
        } catch (e: PackageManager.NameNotFoundException) {
            DeviceLog.exception("Error getting package info", e)
            ""
        }
    }

    /**
     * @return if the device is in a debuggable state
     */
    private fun isAppDebuggable(): Boolean {
        var debuggable = false
        var couldNotGetApplicationInfo = false

        val pm: PackageManager = context.packageManager
        val pkgName: String = context.packageName

        try {
            val appinfo = pm.getApplicationInfo(pkgName, 0)
            if (0 != ApplicationInfo.FLAG_DEBUGGABLE.let {
                    appinfo.flags = appinfo.flags and it; appinfo.flags
                }) {
                debuggable = true
            }
        } catch (e: PackageManager.NameNotFoundException) {
            DeviceLog.exception("Could not find name", e)
            couldNotGetApplicationInfo = true
        }

        if (couldNotGetApplicationInfo) {
            try {
                val pinfo = pm.getPackageInfo(pkgName, PackageManager.GET_SIGNATURES)
                val signatures = pinfo.signatures
                for (signature in signatures) {
                    val cf = CertificateFactory.getInstance(CERTIFICATE_TYPE_X509)
                    val stream = ByteArrayInputStream(signature.toByteArray())
                    val cert = cf.generateCertificate(stream) as X509Certificate
                    debuggable = cert.subjectX500Principal == DEBUG_CERT
                    if (debuggable) break
                }
            } catch (e: PackageManager.NameNotFoundException) {
                DeviceLog.exception("Could not find name", e)
            } catch (e: CertificateException) {
                DeviceLog.exception("Certificate exception", e)
            }
        }
        return debuggable
    }

    /**
     * https://developer.android.com/reference/android/os/ext/SdkExtensions#getExtensionVersion(int)
     *
     * This method is suitable to use in conditional statements to determine whether an API is available and is safe to use.
     *
     * @return the version of the specified extensions
     */
    private fun getExtensionVersion(): Int {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            SdkExtensions.getExtensionVersion(Build.VERSION_CODES.R)
        } else {
            -1
        }
    }

    /**
     * @return if the device has H264 hardware accelerated decoding
     */
    fun hasX264Decoder(): Boolean {
        return selectAllDecodeCodecs(MimeTypes.VIDEO_H264).isNotEmpty()
    }

    /**
     * @return if the device has H265 hardware accelerated decoding
     */
    fun hasX265Decoder(): Boolean {
        return selectAllDecodeCodecs(MimeTypes.VIDEO_H265).isNotEmpty()
    }

    /**
     * @param mimeType of decoders to return
     *
     * @return all decoders available for a specific mimetype
     */
    private fun selectAllDecodeCodecs(mimeType: String?): List<MediaCodecInfo> {
        val result: MutableList<MediaCodecInfo> = mutableListOf()
        val numCodecs = MediaCodecList.getCodecCount()
        for (i in 0 until numCodecs) {
            val codecInfo = MediaCodecList.getCodecInfoAt(i)
            if (codecInfo.isEncoder) {
                continue
            }
            val types = codecInfo.supportedTypes
            for (j in types.indices) {
                if (types[j].equals(mimeType, ignoreCase = true)) {
                    if (isHardwareAccelerated(codecInfo, mimeType)) {
                        result.add(codecInfo)
                    }
                }
            }
        }
        return result
    }

    /**
     * The result of [android.media.MediaCodecInfo.isHardwareAccelerated] for API levels 29+,
     * or a best-effort approximation for lower levels.
     *
     * @return if a [MediaCodecInfo] supports hardware accelerated based decoding
     */
    private fun isHardwareAccelerated(codecInfo: MediaCodecInfo, mimeType: String?): Boolean {
        return if (getApiLevel() >= 29) {
            isHardwareAcceleratedV29(codecInfo)
        } else !isSoftwareOnly(codecInfo, mimeType)
        // codecInfo.isHardwareAccelerated() != codecInfo.isSoftwareOnly() is not necessarily true.
        // However, we assume this to be true as an approximation.
    }

    /**
     * @return if a [MediaCodecInfo] supports hardware accelerated based decoding
     */
    @TargetApi(29)
    private fun isHardwareAcceleratedV29(codecInfo: MediaCodecInfo): Boolean {
        return codecInfo.isHardwareAccelerated
    }

    /**
     * The result of [android.media.MediaCodecInfo.isSoftwareOnly] for API levels 29+, or a
     * best-effort approximation for lower levels.
     *
     * @return if a [MediaCodecInfo] supports only software based decoding
     */
    private fun isSoftwareOnly(codecInfo: MediaCodecInfo, mimeType: String?): Boolean {
        if (getApiLevel() >= 29) {
            return isSoftwareOnlyV29(codecInfo)
        }
        val codecName = codecInfo.name.toLowerCase(Locale.ROOT)
        return if (codecName.startsWith("arc.")) {
            // App Runtime for Chrome (ARC) codecs
            false
        } else ((codecName.startsWith("omx.google.")
            || codecName.startsWith("omx.ffmpeg.")) || codecName.startsWith("omx.sec.") && codecName.contains(
            ".sw."
        ) || codecName == "omx.qcom.video.decoder.hevcswvdec" || codecName.startsWith("c2.android.")
            || codecName.startsWith("c2.google.")) || !codecName.startsWith("omx.") && !codecName.startsWith(
            "c2."
        )
    }

    /**
     * @return if a [MediaCodecInfo] supports only software based decoding
     */
    @TargetApi(29)
    private fun isSoftwareOnlyV29(codecInfo: MediaCodecInfo): Boolean {
        return codecInfo.isSoftwareOnly
    }

    companion object {
        const val PREF_KEY_INSTALLINFO: String = "unityads-installinfo"
        const val PREF_KEY_IDFI: String = "unityads-idfi"
        const val PREF_KEY_SUPERSONIC: String = "supersonic_shared_preferen"
        const val PREF_KEY_AUID: String = "auid"
        const val ALGORITHM_SHA1: String = "SHA-1"
        const val CERTIFICATE_TYPE_X509: String = "X.509"
        const val STORE_GOOGLE: String = "google"
        const val PLATFORM_ANDROID: String = "android"
        const val APP_VERSION_FAKE: String = "FakeVersionName"
        const val BINARY_SU: String = "su"
        const val ENVIRONMENT_VARIABLE_PATH: String = "PATH"
    }
}