package com.unity3d.ads.adplayer

import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.ShowEvent
import gateway.v1.AllowedPiiOuterClass.AllowedPii
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import org.json.JSONObject

/**
 * This interface represents an ad player that can be used to play advertisements.
 * It provides methods to control the player's behavior and events for tracking ad playback.
 **/
interface AdPlayer {
    /**
     * Represents a CoroutineScope object associated to the lifetime of the AdPlayer. It can be used to launch Flows
     * that will be cancelled when the AdPlayer is destroyed.
     **/
    val scope: CoroutineScope

    /**
     * Represents a Flow of ShowEvent objects that can be used to track ad playback.
     **/
    val onShowEvent: Flow<ShowEvent>


    /**
     * Represents the event that is triggered when the AdViewer html has finished loading.
     *
     * Only triggered once per ad view.
     *
     * If an error occurs, it will throw an exception.
     **/
    val loadEvent: Deferred<Unit>
    val updateCampaignState: Flow<Pair<ByteString, Int>>
    val markCampaignStateAsShown: Flow<Unit>

    // region AdViewer API

    /**
     * Sends a request to the AdViewer to start playing the ad.
     **/
    suspend fun requestShow()

    // endregion AdViewer API

    // region Events

    suspend fun sendMuteChange(isMuted: Boolean)
    suspend fun sendVisibilityChange(isVisible: Boolean)
    suspend fun sendVolumeChange(volume: Double)
    suspend fun sendUserConsentChange(value: ByteString)
    suspend fun sendPrivacyFsmChange(value: ByteString)
    suspend fun onAllowedPiiChange(value: AllowedPii)

    suspend fun onBroadcastEvent(event: JSONObject)

    companion object {
        val broadcastEventChannel = MutableSharedFlow<JSONObject>()
    }

    // endregion Events
}