package com.unity3d.ads.core.data.repository

import android.content.pm.ActivityInfo
import com.unity3d.ads.core.domain.AndroidGetLifecycleFlow
import com.unity3d.ads.core.domain.LifecycleEvent
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.filterIsInstance
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.update

/**
 * Repository class that provides mean to track the orientation of the resumed activity.
 */
class OrientationRepository(
    private val getLifecycleFlow: AndroidGetLifecycleFlow,
    private val defaultDispatcher: CoroutineDispatcher,
) {
    private val isRunning = MutableStateFlow(false)

    /**
     * Maintains a state flow that holds a reference to resumed `Activity` orientation.
     *
     * This property is used to track the orientation of the known resumed activity or
     * SCREEN_ORIENTATION_UNSPECIFIED if no activity is tracked.
     */
    private val _resumedActivityOrientation: MutableStateFlow<Int> = MutableStateFlow(ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED)
    val resumedActivityOrientation: StateFlow<Int> = _resumedActivityOrientation

    /**
     * Subscribes to the resumedActivity flow and track their orientation.
     *
     * This method ensures that only a single instance of the tracking logic is active at any time.
     * If invoked while already running, the method will exit without starting a new subscription.
     */
    operator fun invoke() {
        if (isRunning.getAndUpdate { true }) return

        getLifecycleFlow()
            .filterIsInstance<LifecycleEvent.Resumed>()
            .map(LifecycleEvent::activity::get)
            .onEach { activity ->
                _resumedActivityOrientation.value = activity.get()?.requestedOrientation
                        ?: ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED
            }.launchIn(CoroutineScope(defaultDispatcher))
    }
}