package com.unity3d.ads.core.domain.events


import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.core.data.datasource.UniversalRequestDataSource
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.domain.GetRequestPolicy
import com.unity3d.ads.core.domain.GetUniversalRequestForPayLoad
import com.unity3d.ads.core.domain.work.BackgroundWorker
import com.unity3d.ads.core.domain.work.DiagnosticEventJob
import com.unity3d.ads.core.domain.work.UniversalRequestWorkerData
import gatewayprotocol.v1.UniversalRequestKt.payload
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext
import java.util.UUID

class DiagnosticEventObserver(
    private val getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
    private val getDiagnosticEventBatchRequest: GetDiagnosticEventBatchRequest,
    defaultDispatcher: CoroutineDispatcher,
    private val diagnosticEventRepository: DiagnosticEventRepository,
    private val universalRequestDataSource: UniversalRequestDataSource,
    private val backgroundWorker: BackgroundWorker,
    private val universalRequestEventSender: UniversalRequestEventSender,
    private val diagnosticRequestPolicy: GetRequestPolicy
) {
    private val isRunning = MutableStateFlow(false)

    // We provide an exception handler in the worst case scenario where we can't persist nor even send out the event.
    private val scope = CoroutineScope(defaultDispatcher + CoroutineExceptionHandler { _, _ -> })

    suspend operator fun invoke() = withContext(scope.coroutineContext) {
        // If we are already collecting events, do nothing
        if (isRunning.getAndUpdate { true }) return@withContext
        diagnosticEventRepository.diagnosticEvents.onEach { currentDiagnosticEventRequest ->
            val payload = payload {
                diagnosticEventRequest = getDiagnosticEventBatchRequest(currentDiagnosticEventRequest)
            }
            val fullRequest = getUniversalRequestForPayLoad(payload)

            try {
                val workId = UUID.randomUUID().toString()
                universalRequestDataSource.set(workId, fullRequest.toByteArray().toByteString())

                val universalRequestWorkerData = UniversalRequestWorkerData(workId)
                backgroundWorker<DiagnosticEventJob>(universalRequestWorkerData)
            } catch (throwable: Throwable) {
                universalRequestEventSender(fullRequest, diagnosticRequestPolicy())
            }
        }.launchIn(scope)
    }
}