package com.unity3d.ads.core.domain

import com.unity3d.ads.UnityAds
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.model.InitializationState.FAILED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZING
import com.unity3d.ads.core.data.model.InitializationState.NOT_INITIALIZED
import com.unity3d.services.core.configuration.EnvironmentCheck
import com.unity3d.services.core.configuration.InitializeEventsMetricSender
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.properties.SdkProperties
import com.unity3d.services.core.request.metrics.InitMetric

class CommonShouldAllowInitialization(
    val alternativeFlowReader: AlternativeFlowReader,
    val checkForGameIdAndTestModeChanges: CheckForGameIdAndTestModeChanges,
    val getInitializationState: GetInitializationState,
    val setInitializationState: SetInitializationState,
    val validateGameId: ValidateGameId
) : ShouldAllowInitialization {
    override fun invoke(gameId: String?): Boolean {

        if (!validateGameId(gameId)) return false

        if (alternativeFlowReader()) {

            when (getInitializationState(legacy = false)) {
                INITIALIZING -> return false
                INITIALIZED -> {
                    SdkProperties.notifyInitializationComplete()
                    return false
                }
                NOT_INITIALIZED, FAILED -> {
                    checkForGameIdAndTestModeChanges(legacy = false)
                    setInitializationState(INITIALIZING, legacy = false)
                    return true
                }
            }
        } else {
            when (getInitializationState(legacy = true)) {
                INITIALIZING -> return false
                INITIALIZED -> {
                    SdkProperties.notifyInitializationComplete()
                    return false
                }
                FAILED -> {
                    SdkProperties.notifyInitializationFailed(
                        UnityAds.UnityAdsInitializationError.INTERNAL_ERROR,
                        "Unity Ads SDK failed to initialize due to previous failed reason"
                    )
                    return false
                }
                NOT_INITIALIZED ->  {
                    if (!EnvironmentCheck.isEnvironmentOk()) {
                        DeviceLog.error("Error during Unity Services environment check, halting Unity Services init")
                        SdkProperties.notifyInitializationFailed(
                            UnityAds.UnityAdsInitializationError.INTERNAL_ERROR,
                            "Unity Ads SDK failed to initialize due to environment check failed"
                        )
                        InitializeEventsMetricSender.getInstance().sendMetric(InitMetric.newInitEnvironmentNotOk())
                        return false
                    } else {
                        DeviceLog.info("Unity Services environment check OK")
                        InitializeEventsMetricSender.getInstance().sendMetric(InitMetric.newInitEnvironmentOk())
                    }
                    checkForGameIdAndTestModeChanges(legacy = true)
                    setInitializationState(INITIALIZING, legacy = true)
                    return true
                }
            }
        }
    }
}