package com.unity3d.ads.core.data.repository

import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.ATTEMPT_INSERT_NULL_DIAGNOSTIC_EVENT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.NULL_DIAGNOSTIC_EVENT
import com.unity3d.ads.core.domain.events.GetDiagnosticEventRequest
import com.unity3d.ads.core.utils.CoroutineTimer
import com.unity3d.services.core.log.DeviceLog
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticEventType
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticEvent
import gatewayprotocol.v1.NativeConfigurationOuterClass.DiagnosticEventsConfiguration
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import java.util.Collections

internal class AndroidDiagnosticEventRepository(
    private val flushTimer: CoroutineTimer,
    private val getDiagnosticEventRequest: GetDiagnosticEventRequest,
    dispatcher: CoroutineDispatcher,
) : DiagnosticEventRepository {
    private val coroutineScope = CoroutineScope(dispatcher) + CoroutineName("DiagnosticEventRepository")
    private val batch = MutableStateFlow<List<DiagnosticEvent>>(emptyList())
    private var maxBatchSize = Int.MAX_VALUE
    private val allowedEvents = Collections.synchronizedSet(mutableSetOf<DiagnosticEventType>())
    private val blockedEvents = Collections.synchronizedSet(mutableSetOf<DiagnosticEventType>())
    private val enabled = MutableStateFlow(false)
    private val configured = MutableStateFlow(false)

    private val _diagnosticEvents = MutableSharedFlow<List<DiagnosticEvent>>(replay = 100)
    override val diagnosticEvents = _diagnosticEvents.asSharedFlow()

    override fun addDiagnosticEvent(diagnosticEvent: DiagnosticEvent) {
        // It seems that we are getting null on prod. This should never happen. Remove once confirmed.
        val event = diagnosticEvent
            ?: getDiagnosticEventRequest(ATTEMPT_INSERT_NULL_DIAGNOSTIC_EVENT, null, null, null, null, null, null, null, null)

        if (!configured.value) {
            batch.update { it + event }
        } else {
            if (!enabled.value) return
            batch.update { it + event }
            if (batch.value.size >= maxBatchSize) {
                flush()
            }
        }
    }

    override fun flush() {
        if (!enabled.value) return
        val events = batch.getAndUpdate { emptyList() }
            .asSequence<DiagnosticEvent?>()
            // It seems that we are getting null on prod. This should never happen. Remove once confirmed.
            .map { it ?: getDiagnosticEventRequest(NULL_DIAGNOSTIC_EVENT, null, null, null, null, null, null, null, null) }
            .filter { allowedEvents.isEmpty() || it.eventType in allowedEvents }
            .filter { it.eventType !in blockedEvents }
            .toList()

        if (events.isNotEmpty()) {
            DeviceLog.debug {
                "Unity Ads Sending diagnostic batch enabled: ${enabled.value} size: ${events.size} :: $events"
            }
            coroutineScope.launch {
                _diagnosticEvents.emit(events)
            }
        }
    }

    override fun clear() {
        batch.update { emptyList() }
    }

    override fun configure(diagnosticsEventsConfiguration: DiagnosticEventsConfiguration) {
        configured.value = true
        enabled.value = diagnosticsEventsConfiguration.enabled
        if (!enabled.value) {
            clear()
            return
        }
        this.maxBatchSize = diagnosticsEventsConfiguration.maxBatchSize
        this.allowedEvents.addAll(diagnosticsEventsConfiguration.allowedEventsList)
        this.blockedEvents.addAll(diagnosticsEventsConfiguration.blockedEventsList)

        val maxBatchIntervalMs = diagnosticsEventsConfiguration.maxBatchIntervalMs

        flushTimer.start(
            delayStartMillis = 0,
            repeatMillis = maxBatchIntervalMs.toLong(),
        ) { flush() }
    }
}