package com.unity3d.services.core.device

import com.unity3d.services.core.device.StorageManager.StorageType
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.misc.JsonStorage
import com.unity3d.services.core.misc.Utilities
import com.unity3d.services.core.webview.WebViewApp
import com.unity3d.services.core.webview.WebViewEventCategory
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update
import org.json.JSONObject
import java.io.File
import java.io.FileNotFoundException

data class StorageEventInfo(
    val eventType: StorageEvent,
    val storageType: StorageType,
    val value: Any?
)

open class Storage(private val _targetFileName: String, val type: StorageType) : JsonStorage() {
    @Synchronized
    open fun readStorage(): Boolean {
        val f = File(_targetFileName)
        return try {
            val bytes = Utilities.readFileBytes(f) ?: return false
            val fileData = String(bytes)
            data = JSONObject(fileData)
            true
        } catch (e: FileNotFoundException) {
            DeviceLog.debug("Storage JSON file not found in local cache:", e)
            false
        } catch (e: Exception) {
            DeviceLog.debug("Failed to read storage JSON file:", e)
            false
        }
    }

    @Synchronized
    fun initStorage(): Boolean {
        readStorage()
        super.initData()
        return true
    }

    @Synchronized
    open fun writeStorage(): Boolean {
        val f = File(_targetFileName)
        if (data != null) {
            val success: Boolean
            success = Utilities.writeFile(f, data.toString())
            return success
        }
        return false
    }

    @Synchronized
    open fun clearStorage(): Boolean {
        clearData()
        val f = File(_targetFileName)
        return f.delete()
    }

    @Synchronized
    fun storageFileExists(): Boolean {
        val f = File(_targetFileName)
        return f.exists()
    }

    @Synchronized
    fun sendEvent(eventType: StorageEvent?, value: Any?) {
        val callbacks = onStorageEventCallbacks.value
        if (callbacks.isNotEmpty()) {
            val event = StorageEventInfo(eventType!!, type, value)

            callbacks.forEach { it(event) }
            return
        }

        var success = false
        if (WebViewApp.getCurrentApp() != null) {
            success = WebViewApp.getCurrentApp().sendEvent(WebViewEventCategory.STORAGE, eventType, type.name, value)
        }
        if (!success) {
            DeviceLog.debug("Couldn't send storage event to WebApp")
        }
    }



    companion object {
        private val onStorageEventCallbacks = MutableStateFlow(emptyList<(StorageEventInfo) -> Unit>())

        fun addStorageEventCallback(callback: (StorageEventInfo) -> Unit) = onStorageEventCallbacks.update { it + callback }
        fun removeStorageEventCallback(callback: (StorageEventInfo) -> Unit) = onStorageEventCallbacks.update { it - callback }
    }
}
