package com.unity3d.ads.core.domain.billing

import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.billingclient.BillingClientAdapter
import com.unity3d.services.store.gpbl.listeners.ProductDetailsListener
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume

/**
 * Fetches product details for a specific product type (INAPP or SUBS).
 */
class CommonProductDetailsFetcher(
    private val billingClientAdapter: BillingClientAdapter?,
    private val productType: String
) : ProductDetailsFetcher {
    
    override suspend fun fetchProductDetails(
        productId: String
    ): ProductDetailsResult {
        return suspendCancellableCoroutine { continuation ->
            val listener = ProductDetailsListener { billingResult, productDetailsJson ->
                when {
                    productDetailsJson != null && billingResult.responseCode == BillingResultResponseCode.OK -> {
                        continuation.resume(ProductDetailsResult.Success(productDetailsJson))
                    }
                    billingResult.responseCode == BillingResultResponseCode.OK -> {
                        continuation.resume(ProductDetailsResult.NotFound)
                    }
                    else -> {
                        continuation.resume(ProductDetailsResult.Failure(billingResult))
                    }
                }
            }
            
            billingClientAdapter?.queryProductDetailsAsync(
                productId = productId,
                productType = productType,
                listener = listener
            )
        }
    }
}


