package com.unity3d.ads.core.domain

import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.WebViewConfiguration
import com.unity3d.ads.core.data.repository.CacheRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import java.io.File
import java.util.concurrent.ConcurrentHashMap

/**
 * Implementation of [CacheWebViewAssets] that uses [CacheRepository] to cache WebView assets
 */
internal class AndroidCacheWebViewAssets(
    private val cacheRepository: CacheRepository,
    private val sessionRepository: SessionRepository,
) : CacheWebViewAssets {
    private val _cached = ConcurrentHashMap<String, File>()
    override val cached: Map<String, File>
        get() = _cached

    /**
     * Caches the WebView assets from the provided WebViewConfiguration
     *
     * @param webviewConfiguration The WebView configuration containing the entry point and additional files to cache
     * @param type The type of WebView (e.g., "iAds", "uAds")
     */
    override suspend fun invoke(webviewConfiguration: WebViewConfiguration) {
        // Check if webview caching is enabled in NativeConfiguration
        if (!sessionRepository.nativeConfiguration.featureFlags.shouldHandleWebviewCaching) {
            return
        }

        // Cache the entry point if it's not empty
        if (webviewConfiguration.entryPoint.isNotEmpty()) {
            val result = cacheRepository.getWebviewFile(webviewConfiguration.entryPoint, webviewConfiguration.type)
            if (result is CacheResult.Success && result.cachedFile.file != null) {
                val filename = getFilename(webviewConfiguration.entryPoint, webviewConfiguration.type)
                _cached[filename] = result.cachedFile.file
            }
        }

        // Cache all additional files
        webviewConfiguration.additionalFiles.forEach { additionalFile ->
            if (additionalFile.isNotEmpty()) {
                val result = cacheRepository.getWebviewFile(additionalFile, webviewConfiguration.type)
                if (result is CacheResult.Success && result.cachedFile.file != null) {
                    val filename = getFilename(additionalFile, webviewConfiguration.type)
                    _cached[filename] = result.cachedFile.file
                }
            }
        }
    }

    private fun getFilename(url: String, type: String): String =
        "$type/${url.substringBefore('?').substringAfterLast('/')}"
}
