package com.unity3d.services.core.network.core

import com.unity3d.ads.core.data.model.exception.NetworkTimeoutException
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.network.mapper.toOkHttpProtoRequest
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Response
import java.io.IOException
import java.net.SocketTimeoutException
import java.util.concurrent.TimeUnit
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

/**
 * An implementation of [HttpClient] based on OkHttp
 * Supports Http2
 */
class RefactoredOkHttp3Client(
    private val dispatchers: ISDKDispatchers,
    private val client: OkHttpClient,
) : HttpClient {

    /**
     * Helper method that blocks the thread to be used for Java interaction
     *
     * @param request [HttpRequest] to be executes on the network
     * @return [HttpResponse] of the passed in [HttpRequest]
     */
    override fun executeBlocking(request: HttpRequest): HttpResponse = runBlocking(dispatchers.io) {
        execute(request)
    }

    /**
     * Executes an http network request
     *
     * @param request [HttpRequest] to be executes on the network
     * @return [HttpResponse] of the passed in [HttpRequest]
     */
    override suspend fun execute(request: HttpRequest): HttpResponse {
        try {
            val okHttpRequest = request.toOkHttpProtoRequest()

            val configuredClient = client.newBuilder()
                .connectTimeout(request.connectTimeout.toLong(), TimeUnit.MILLISECONDS)
                .readTimeout(request.readTimeout.toLong(), TimeUnit.MILLISECONDS)
                .writeTimeout(request.writeTimeout.toLong(), TimeUnit.MILLISECONDS)
                .build()

            return suspendCancellableCoroutine { continuation ->
                val call = configuredClient.newCall(okHttpRequest)

                continuation.invokeOnCancellation {
                    call.cancel()
                }

                call.enqueue(object : Callback {
                    override fun onResponse(call: Call, response: Response) {
                        if (!response.isSuccessful) {
                            return continuation.resumeWithException(UnityAdsNetworkException(
                                "Network request failed with code ${response.code()}",
                                code = response.code(),
                                client = NETWORK_CLIENT_OKHTTP,
                            ))
                        }
                        try {
                            val body = response.body() ?: return continuation.resumeWithException(UnityAdsNetworkException(
                                "Empty response",
                                code = response.code(),
                                client = NETWORK_CLIENT_OKHTTP,
                            ))

                            continuation.resume(HttpResponse(
                                statusCode = response.code(),
                                headers = response.headers().toMultimap(),
                                urlString = response.request().url().toString(),
                                body = body.source().readByteArray(),
                                protocol = response.protocol().toString(),
                                client = NETWORK_CLIENT_OKHTTP
                            ))
                        } catch (e: Exception) {
                            continuation.resumeWithException(e)
                        }
                    }

                    override fun onFailure(call: Call, e: IOException) {
                        continuation.resumeWithException(e)
                    }
                })
            }
        } catch (e: SocketTimeoutException) {
            throw NetworkTimeoutException(
                message = MSG_CONNECTION_TIMEOUT,
                url = request.baseURL,
                client = NETWORK_CLIENT_OKHTTP
            )
        } catch (e: IOException) {
            throw UnityAdsNetworkException(
                message = MSG_CONNECTION_FAILED,
                url = request.baseURL,
                client = NETWORK_CLIENT_OKHTTP
            )
        }
    }

    companion object {
        const val MSG_CONNECTION_TIMEOUT = "Network request timeout"
        const val MSG_CONNECTION_FAILED = "Network request failed"
        const val NETWORK_CLIENT_OKHTTP = "refactored-okhttp"
    }
}
