package com.unity3d.ads.core.domain

import android.app.Activity
import android.app.Application
import android.content.Context
import android.os.Bundle
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.channelFlow
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference

sealed interface LifecycleEvent {
    val activity: WeakReference<Activity>

    data class Created(override val activity: WeakReference<Activity>, val bundle: Bundle?): LifecycleEvent
    data class Started(override val activity: WeakReference<Activity>): LifecycleEvent
    data class Resumed(override val activity: WeakReference<Activity>): LifecycleEvent
    data class Paused(override val activity: WeakReference<Activity>): LifecycleEvent
    data class Stopped(override val activity: WeakReference<Activity>): LifecycleEvent
    data class SaveInstanceState(override val activity: WeakReference<Activity>, val bundle: Bundle?): LifecycleEvent
    data class Destroyed(override val activity: WeakReference<Activity>): LifecycleEvent
}

class AndroidGetLifecycleFlow(
    private val applicationContext: Context
) {
    operator fun invoke(): Flow<LifecycleEvent> {
        require(applicationContext is Application) { "Application context is required" }

        return channelFlow {
            val listener = object : Application.ActivityLifecycleCallbacks {
                override fun onActivityCreated(activity: Activity, bundle: Bundle?) {
                    launch { send(LifecycleEvent.Created(WeakReference(activity), bundle)) }
                }

                override fun onActivityStarted(activity: Activity) {
                    launch { send(LifecycleEvent.Started(WeakReference(activity))) }
                }

                override fun onActivityResumed(activity: Activity) {
                    launch { send(LifecycleEvent.Resumed(WeakReference(activity))) }
                }

                override fun onActivityPaused(activity: Activity) {
                    launch { send(LifecycleEvent.Paused(WeakReference(activity))) }
                }

                override fun onActivityStopped(activity: Activity) {
                    launch { send(LifecycleEvent.Stopped(WeakReference(activity))) }
                }

                override fun onActivitySaveInstanceState(activity: Activity, bundle: Bundle) {
                    launch { send(LifecycleEvent.SaveInstanceState(WeakReference(activity), bundle)) }
                }

                override fun onActivityDestroyed(activity: Activity) {
                    launch { send(LifecycleEvent.Destroyed(WeakReference(activity))) }
                }
            }

            applicationContext.registerActivityLifecycleCallbacks(listener)

            awaitClose { applicationContext.unregisterActivityLifecycleCallbacks(listener) }
        }
    }
}