package com.unity3d.ads.core.data.repository

import android.content.Context
import com.unity3d.ads.core.data.datasource.CacheDataSource
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.CreateFile
import com.unity3d.ads.core.domain.GetCacheDirectory
import com.unity3d.ads.core.domain.work.DownloadPriorityQueue
import com.unity3d.ads.core.extensions.getSHA256Hash
import com.unity3d.services.UnityAdsConstants.DefaultUrls.CACHE_DIR_NAME
import com.unity3d.services.UnityAdsConstants.DefaultUrls.CACHE_WEBVIEW_DIR_NAME
import com.unity3d.services.core.network.domain.CleanupDirectory
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.plus
import kotlinx.coroutines.withContext
import org.json.JSONArray
import java.io.File

class AndroidCacheRepository(
    ioDispatcher: CoroutineDispatcher,
    private val getCacheDirectory: GetCacheDirectory,
    private val localCacheDataSource: CacheDataSource,
    private val remoteCacheDataSource: CacheDataSource,
    private val context: Context,
    private val sessionRepository: SessionRepository,
    private val cleanupDirectory: CleanupDirectory,
    private val downloadPriorityQueue: DownloadPriorityQueue,
    private val createFile: CreateFile
) : CacheRepository {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("CacheRepository") + NonCancellable
    private val cacheDir: File = initCacheDir(CACHE_DIR_NAME)
    private val webviewCacheDir: File = initCacheDir(CACHE_WEBVIEW_DIR_NAME)

    override suspend fun getFile(
        url: String,
        headers: JSONArray?,
        priority: Int
    ): CacheResult = getFileInternal(cacheDir, url, headers, priority)

    /**
     * While [getFile] is used for assets, this method is used to handle files that are requires by the Android WebView
     * component.
     *
     * @param url Is either a link from
     * @param type Correspond to the type provided by [gatewayprotocol.v1.WebviewConfiguration.WebViewConfiguration.getType]
     */
    override suspend fun getWebviewFile(url: String, type: String): CacheResult {
        val typeSpecificDir = createFile(webviewCacheDir, type)
        typeSpecificDir.mkdirs()
        return getFileInternal(typeSpecificDir, url, null, 0)
    }

    private suspend fun getFileInternal(
        cacheDirectory: File,
        url: String,
        headers: JSONArray?,
        priority: Int
    ): CacheResult = withContext(scope.coroutineContext) {
        val filename = getFilename(url)

        // Check if file is already in cache
        val localFile = localCacheDataSource.getFile(cacheDirectory, filename, url, priority)
        if (localFile is CacheResult.Success) {
            return@withContext localFile
        }

        // Download the file
        val fileResult = MutableStateFlow<CacheResult?>(null)
        downloadPriorityQueue(priority) {
            // Try local cache first
            val localFilePreemptive = localCacheDataSource.getFile(cacheDirectory, filename, url, priority)
            if (localFilePreemptive is CacheResult.Success) {
                return@downloadPriorityQueue fileResult.update { localFilePreemptive }
            }

            // Try remote cache if local cache fails
            val remoteFile = remoteCacheDataSource.getFile(cacheDirectory, filename, url, priority)
            fileResult.update { remoteFile }
        }

        return@withContext fileResult.filterNotNull().first()
    }

    override suspend fun retrieveFile(fileName: String): CacheResult {
        return localCacheDataSource.getFile(cacheDir, fileName)
    }

    override fun removeFile(cachedFile: CachedFile): Boolean {
        return cachedFile.file?.takeIf(File::exists)?.delete() ?: false
    }

    override suspend fun doesFileExist(fileName: String): Boolean = retrieveFile(fileName) is CacheResult.Success

    fun getFilename(url: String): String = "${url.getSHA256Hash()}.${url.substringAfterLast('.')}"

    override suspend fun clearCache() {
        withContext(scope.coroutineContext) {
            if (!sessionRepository.nativeConfiguration.hasCachedAssetsConfiguration()) {
                return@withContext cacheDir.listFiles()?.forEach(File::delete)
            } else {
                val config = sessionRepository.nativeConfiguration.cachedAssetsConfiguration
                cleanupDirectory(cacheDir, config.maxCachedAssetSizeMb, config.maxCachedAssetAgeMs)
            }

            if (!sessionRepository.nativeConfiguration.hasCachedWebviewFilesConfiguration()) {
                return@withContext webviewCacheDir.listFiles()?.forEach(File::delete)
            } else {
                val webViewConfig = sessionRepository.nativeConfiguration.cachedWebviewFilesConfiguration
                cleanupDirectory(webviewCacheDir, webViewConfig.maxCachedAssetSizeMb, webViewConfig.maxCachedAssetAgeMs)
            }
        }
    }

    override suspend fun getCacheSize(): Long = withContext(scope.coroutineContext) {
        cacheDir.walk().filter(File::isFile).sumOf(File::length)
    }

    private fun initCacheDir(dirName: String): File {
        val dir = getCacheDirectory(context.cacheDir, dirName)
        dir.mkdirs()
        return dir
    }
}
