package com.unity3d.ads.adplayer

import android.view.ViewGroup
import androidx.core.view.doOnAttach
import androidx.core.view.doOnDetach
import com.google.protobuf.kotlin.toByteStringUtf8
import com.unity3d.ads.adplayer.AdPlayer.Companion.SCAR_EVENT_QUEUE_SIZE
import com.unity3d.ads.core.data.manager.ScarManager
import com.unity3d.ads.core.data.model.ScarEvent
import com.unity3d.ads.core.data.repository.OpenMeasurementRepository
import com.unity3d.scar.adapter.common.scarads.ScarAdMetadata
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_SESSION_FINISH_DELAY_MS
import com.unity3d.services.banners.BannerViewCache
import com.unity3d.services.core.misc.ViewUtilities
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.shareIn
import kotlinx.coroutines.launch

internal class AndroidEmbeddableWebViewAdPlayer(
    private val webViewAdPlayer: WebViewAdPlayer,
    private val opportunityId: String,
    override val webViewContainer: AndroidWebViewContainer,
    private val openMeasurementRepository: OpenMeasurementRepository,
    private val scarManager: ScarManager
) : AdPlayer by webViewAdPlayer, EmbeddableAdPlayer {
    override fun show(showOptions: ShowOptions) {
        require(showOptions is AndroidShowOptions)
        val context = showOptions.activity.get()
        requireNotNull(context)
        val bannerView = checkNotNull(BannerViewCache.getInstance().getBannerView(opportunityId)) {
            "BannerView not found for opportunityId: $opportunityId"
        }

        if (showOptions.isScarAd) {
            val scarAdMetadata = ScarAdMetadata(
                showOptions.placementId ?: "",
                showOptions.scarQueryId ?: "",
                showOptions.scarAdUnitId ?: "",
                showOptions.scarAdString ?: "",
                0
            )

            /**
             * SCAR / AdPlayer "event dance" flow:
             * 1. Request show from SCAR and start collecting events
             * 2. Let AdPlayer know that SCAR is ready to show
             * 3. Collect SCAR events and map them to GMA events when AdPlayer is ready to receive them
             */
            val scarEvents = scarManager.loadBannerAd(
                context = context,
                bannerView = bannerView,
                scarAdMetadata = scarAdMetadata,
                bannerSize = bannerView.size,
                opportunityId = opportunityId
            ).shareIn(scope, SharingStarted.Eagerly, SCAR_EVENT_QUEUE_SIZE)

            bannerView.doOnAttach {
                scope.launch {
                    onScarEvent
                        .onStart {
                            webViewAdPlayer.requestShow(showOptions.unityAdsShowOptions)
                        }
                        .first { it == ScarEvent.Show }

                    scarEvents
                        .mapNotNull { it.bannerEvent }
                        .collect(webViewAdPlayer::sendScarBannerEvent)
                }
                bannerView.doOnDetach {
                    webViewAdPlayer.scope.launch {
                        destroy()
                    }
                }
            }
        } else {
            MainScope().launch {
                webViewContainer.webView.layoutParams = ViewGroup.LayoutParams(
                    ViewUtilities.pxFromDp(context, bannerView.size.width.toFloat()).toInt(),
                    ViewUtilities.pxFromDp(context, bannerView.size.height.toFloat()).toInt(),
                )
            }

            webViewContainer.webView.doOnAttach {
                webViewAdPlayer.scope.launch {
                    webViewAdPlayer.requestShow(showOptions.unityAdsShowOptions)
                }

                it.doOnDetach {
                    webViewAdPlayer.scope.launch {
                        destroy()
                    }
                }
            }

            MainScope().launch {
                bannerView.addView(webViewContainer.webView)
            }
        }


    }

    override suspend fun destroy() {
        webViewAdPlayer.dispatchShowCompleted()

        // If the if had an OMID session we need to delay webview cleanup for 1 second.
        // https://interactiveadvertisingbureau.github.io/Open-Measurement-SDKAndroid/#9-stop-the-session
        if (openMeasurementRepository.hasSessionFinished(opportunityId.toByteStringUtf8())) {
            delay(OM_SESSION_FINISH_DELAY_MS)
        }
        // Destroy the webview container.
        webViewContainer.destroy()

        super<AdPlayer>.destroy()
    }
}