package com.unity3d.services.core.di

import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.datastore.core.DataMigration
import androidx.datastore.core.DataStore
import androidx.datastore.core.DataStoreFactory
import androidx.datastore.dataStoreFile
import com.google.android.gms.net.CronetProviderInstaller
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.datasource.AndroidByteStringDataSource
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.ByteStringSerializer
import com.unity3d.ads.core.data.model.UniversalRequestStoreSerializer
import com.unity3d.ads.core.data.model.WebViewConfigurationStoreSerializer
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_SUCCESS
import com.unity3d.ads.datastore.ByteStringStoreOuterClass.ByteStringStore
import com.unity3d.ads.datastore.UniversalRequestStoreOuterClass
import com.unity3d.ads.datastore.WebviewConfigurationStore
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.UnityAdsConstants.DefaultUrls.HTTP_CACHE_DIR_NAME
import com.unity3d.services.ads.measurements.MeasurementsService
import com.unity3d.services.ads.token.AsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryAsyncTokenStorage
import com.unity3d.services.ads.token.TokenStorage
import com.unity3d.services.ads.topics.TopicsService
import com.unity3d.services.core.device.StorageManager
import com.unity3d.services.core.device.StorageManager.StorageType
import com.unity3d.services.core.device.VolumeChange
import com.unity3d.services.core.device.VolumeChangeMonitor
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_HOST
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_PORT
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GATEWAY_CACHE
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GL_INFO
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IAP_TRANSACTION
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_NATIVE_CONFIG
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY_FSM
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_UNIVERSAL_REQUEST
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_WEBVIEW_CONFIG
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_HOST
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_PORT
import com.unity3d.services.core.di.ServiceProvider.HTTP_CACHE_DISK_SIZE
import com.unity3d.services.core.di.ServiceProvider.HTTP_CLIENT_FETCH_TIMEOUT
import com.unity3d.services.core.di.ServiceProvider.NAMED_GET_TOKEN_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_LOAD_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_OMID_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_SHOW_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_TRANSACTION_SCOPE
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.domain.SDKDispatchers
import com.unity3d.services.core.domain.task.ConfigFileFromLocalStorage
import com.unity3d.services.core.misc.JsonStorage
import com.unity3d.services.core.network.core.CronetClient
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.core.LegacyHttpClient
import com.unity3d.services.core.network.core.OkHttp3Client
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.request.metrics.SDKMetrics
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import com.unity3d.services.core.webview.bridge.SharedInstances
import gatewayprotocol.v1.NativeConfigurationOuterClass.AdOperationsConfiguration
import gatewayprotocol.v1.NativeConfigurationOuterClass.NativeConfiguration
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestPolicy
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestRetryPolicy
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestTimeoutPolicy
import gatewayprotocol.v1.adOperationsConfiguration
import gatewayprotocol.v1.diagnosticEventsConfiguration
import gatewayprotocol.v1.nativeConfiguration
import gatewayprotocol.v1.requestPolicy
import gatewayprotocol.v1.requestRetryPolicy
import gatewayprotocol.v1.requestTimeoutPolicy
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import okhttp3.OkHttpClient
import org.chromium.net.CronetEngine
import kotlin.coroutines.resume
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

class UnityAdsModule {
    fun androidContext(): Context = ClientProperties.getApplicationContext()

    fun mainDispatcher(): CoroutineDispatcher = Dispatchers.Main

    fun defaultDispatcher(): CoroutineDispatcher = Dispatchers.Default

    fun ioDispatcher(): CoroutineDispatcher = Dispatchers.IO

    fun sdkDispatchers(): ISDKDispatchers = SDKDispatchers()

    fun sdkMetrics(): SDKMetricsSender = SDKMetrics.getInstance()

    fun initCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_INIT_SCOPE) + errorHandler)

    fun loadCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_LOAD_SCOPE) + errorHandler)

    fun showCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_SHOW_SCOPE) + errorHandler)

    fun transactionCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope =
        CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_TRANSACTION_SCOPE) + errorHandler)

    fun getTokenCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope =
        CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_GET_TOKEN_SCOPE) + errorHandler)

    fun omidCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_OMID_SCOPE) + errorHandler)

    fun publicApiJob(
        diagnosticEventRepository: DiagnosticEventRepository,
    ): Job = Job().apply {
        invokeOnCompletion {
            diagnosticEventRepository.flush()
        }
    }

    fun gatewayDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_GATEWAY_CACHE)

    fun privacyDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY)

    fun privacyFsmDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY_FSM)

    fun nativeConfigurationDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_NATIVE_CONFIG)

    fun glInfoDataStore(
        context: Context, dispatcher: CoroutineDispatcher, fetchGLInfo: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> = DataStoreFactory.create(
        serializer = ByteStringSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_GL_INFO) },
        migrations = listOf(fetchGLInfo),
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    fun universalRequestDataStore(
        context: Context, dispatcher: CoroutineDispatcher
    ): DataStore<UniversalRequestStoreOuterClass.UniversalRequestStore> = DataStoreFactory.create(
        serializer = UniversalRequestStoreSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_UNIVERSAL_REQUEST) },
        corruptionHandler = null,
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    fun iapTransactionDataStore(
        context: Context, dispatcher: CoroutineDispatcher
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_IAP_TRANSACTION)

    fun webViewConfigurationDataStore(
        context: Context, dispatcher: CoroutineDispatcher
    ): DataStore<WebviewConfigurationStore.WebViewConfigurationStore> = DataStoreFactory.create(
        serializer = WebViewConfigurationStoreSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_WEBVIEW_CONFIG) },
        corruptionHandler = null,
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    fun asyncTokenStorage(
        tokenStorage: TokenStorage,
        sdkMetricsSender: SDKMetricsSender,
    ): AsyncTokenStorage = InMemoryAsyncTokenStorage(
        null, Handler(Looper.getMainLooper()), sdkMetricsSender, tokenStorage
    )

    fun volumeChangeMonitor(
        volumeChange: VolumeChange,
    ): VolumeChangeMonitor = VolumeChangeMonitor(SharedInstances.webViewEventSender, volumeChange)

    fun publicJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PUBLIC)

    fun privateJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PRIVATE)

    fun defaultNativeConfiguration(): NativeConfiguration = nativeConfiguration {
        adOperations = getDefaultAdOperations()
        initPolicy = getDefaultRequestPolicy()
        adPolicy = getDefaultRequestPolicy()
        otherPolicy = getDefaultRequestPolicy()
        operativeEventPolicy = getDefaultRequestPolicy()
        diagnosticEvents = diagnosticEventsConfiguration {
            enabled = true
            maxBatchSize = 10
            maxBatchIntervalMs = 30000
            ttmEnabled = false
        }
    }

    fun gatewayCacheDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun privacyDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun idfiDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun auidDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun privacyFsmDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun nativeConfigurationDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun glInfoDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun iapTransactionDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun measurementService(
        context: Context,
        dispatchers: ISDKDispatchers,
    ): MeasurementsService = MeasurementsService(context, dispatchers, SharedInstances.webViewEventSender)

    fun topicsService(
        context: Context, dispatchers: ISDKDispatchers
    ): TopicsService = TopicsService(context, dispatchers, SharedInstances.webViewEventSender)

    @OptIn(ExperimentalTime::class)
    fun provideHttpClient(
        configFileFromLocalStorage: ConfigFileFromLocalStorage,
        alternativeFlowReader: AlternativeFlowReader,
        dispatchers: ISDKDispatchers,
        sendDiagnosticEvent: SendDiagnosticEvent,
        context: Context
    ): HttpClient = runBlocking {
        val isAlternativeFlowEnabled = alternativeFlowReader()
        if (isAlternativeFlowEnabled) {
            val startTime = TimeSource.Monotonic.markNow()
            val client = withTimeoutOrNull(HTTP_CLIENT_FETCH_TIMEOUT) {
                buildNetworkClient(context, dispatchers)
            }
            val diagnosticResult = if (client == null) SYSTEM_CRONET_FAILURE else SYSTEM_CRONET_SUCCESS
            sendDiagnosticEvent(diagnosticResult, startTime.elapsedNow().toDouble(DurationUnit.MILLISECONDS))
            client ?: OkHttp3Client(dispatchers, OkHttpClient())
        } else {
            val config = runBlocking {
                runCatching { configFileFromLocalStorage(ConfigFileFromLocalStorage.Params()) }.getOrNull()?.getOrNull()
            }
            if (config?.experiments?.isOkHttpEnabled == true) {
                OkHttp3Client(dispatchers, OkHttpClient())
            } else {
                LegacyHttpClient(dispatchers)
            }
        }
    }

    private suspend fun buildNetworkClient(
        context: Context, dispatchers: ISDKDispatchers
    ): HttpClient = suspendCancellableCoroutine { continuation ->
        CronetProviderInstaller.installProvider(context).addOnCompleteListener {
            if (it.isSuccessful) {
                try {
                    val cronetEngine =
                        CronetEngine.Builder(context).setStoragePath(buildCronetCachePath(context))
                            .enableHttpCache(CronetEngine.Builder.HTTP_CACHE_DISK, HTTP_CACHE_DISK_SIZE) // HTTP_CACHE_DISK provides 0-RTT support
                            .enableQuic(true)
                            .addQuicHint(GATEWAY_HOST, GATEWAY_PORT, GATEWAY_PORT)
                            .addQuicHint(CDN_CREATIVES_HOST, CDN_CREATIVES_PORT, CDN_CREATIVES_PORT)
                            .build()
                    continuation.resume(CronetClient(cronetEngine, dispatchers))
                } catch (e: Throwable) {
                    continuation.resume(OkHttp3Client(dispatchers, OkHttpClient()))
                }
            } else {
                continuation.resume(OkHttp3Client(dispatchers, OkHttpClient()))
            }
        }
    }

    private fun buildCronetCachePath(context: Context): String {
        // https://chromium.googlesource.com/chromium/src/+/lkgr/components/cronet/android/java/src/org/chromium/net/impl/CronetEngineBuilderImpl.java#238
        // this directory will also be fully wiped by Cronet
        val cacheDir = context.filesDir.resolve(HTTP_CACHE_DIR_NAME)
        if (!cacheDir.exists()) {
            cacheDir.mkdirs()
        }
        return cacheDir.absolutePath
    }

    private fun provideJsonStorage(storageType: StorageType): JsonStorage {
        check(StorageManager.init(ClientProperties.getApplicationContext())) {
            "StorageManager failed to initialize"
        }
        return StorageManager.getStorage(storageType)
    }

    private fun provideByteStringDataSource(dataStore: DataStore<ByteStringStore>): ByteStringDataSource {
        return AndroidByteStringDataSource(dataStore)
    }

    private fun provideByteStringDataStore(
        context: Context, dispatcher: CoroutineDispatcher, dataStoreFile: String
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(dataStoreFile) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatcher + SupervisorJob())
        )
    }

    private fun getDefaultAdOperations(): AdOperationsConfiguration {
        return adOperationsConfiguration {
            loadTimeoutMs = UnityAdsConstants.AdOperations.LOAD_TIMEOUT_MS
            showTimeoutMs = UnityAdsConstants.AdOperations.SHOW_TIMEOUT_MS
            getTokenTimeoutMs = UnityAdsConstants.AdOperations.GET_TOKEN_TIMEOUT_MS
        }
    }

    private fun getDefaultRequestPolicy(): RequestPolicy {
        return requestPolicy {
            retryPolicy = getDefaultRequestRetryPolicy()
            timeoutPolicy = getDefaultRequestTimeoutPolicy()
        }
    }

    private fun getDefaultRequestRetryPolicy(): RequestRetryPolicy {
        return requestRetryPolicy {
            maxDuration = UnityAdsConstants.RequestPolicy.RETRY_MAX_DURATION
            retryWaitBase = UnityAdsConstants.RequestPolicy.RETRY_WAIT_BASE
            retryJitterPct = UnityAdsConstants.RequestPolicy.RETRY_JITTER_PCT
            shouldStoreLocally = UnityAdsConstants.RequestPolicy.SHOULD_STORE_LOCALLY
            retryMaxInterval = UnityAdsConstants.RequestPolicy.RETRY_MAX_INTERVAL
            retryScalingFactor = UnityAdsConstants.RequestPolicy.RETRY_SCALING_FACTOR
        }
    }

    private fun getDefaultRequestTimeoutPolicy(): RequestTimeoutPolicy {
        return requestTimeoutPolicy {
            connectTimeoutMs = UnityAdsConstants.RequestPolicy.CONNECT_TIMEOUT_MS
            readTimeoutMs = UnityAdsConstants.RequestPolicy.READ_TIMEOUT_MS
            writeTimeoutMs = UnityAdsConstants.RequestPolicy.WRITE_TIMEOUT_MS
            overallTimeoutMs = UnityAdsConstants.RequestPolicy.OVERALL_TIMEOUT_MS
        }
    }
}
