package com.unity3d.ads.core.domain

import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_STARTED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_SUCCESS
import com.unity3d.ads.UnityAds.UnityAdsInitializationError.INTERNAL_ERROR
import com.unity3d.ads.core.domain.om.InitializeOMSDK
import com.unity3d.ads.core.data.manager.SDKPropertiesManager
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.domain.events.EventObservers
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.ads.core.data.manager.StorageManager
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.exception.InitializationException
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.configuration.ConfigurationReader
import com.unity3d.services.core.di.ServiceProvider.DEFAULT_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_REQ
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeout


import kotlin.time.ExperimentalTime
import kotlin.time.TimeMark
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)

internal class InitializeAndroidBoldSDK(
    private val defaultDispatcher: CoroutineDispatcher,
    private val initializeOM: InitializeOMSDK,
    private val getInitializeRequest: GetInitializationRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val clearCache: ClearCache,
    private val handleGatewayInitializationResponse: HandleGatewayInitializationResponse,
    private val gatewayClient: GatewayClient,
    private val sessionRepository: SessionRepository,
    private val eventObservers: EventObservers,
    private val triggerInitializeListener: TriggerInitializeListener,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val diagnosticEventRepository: DiagnosticEventRepository,
    private val storageManager: StorageManager,
    private val legacyConfigurationReader: ConfigurationReader,
    private val sdkPropertiesManager: SDKPropertiesManager,
) : InitializeBoldSDK {

    override suspend fun invoke() = withContext(defaultDispatcher) {
        val startTime = TimeSource.Monotonic.markNow()

        try {
            withTimeout(UnityAdsConstants.Timeout.INIT_TIMEOUT_MS) {
                initializationStart()
                checkCanInitialize()
                val initializationRequest = getInitializeRequest()
                val requestPolicy = getRequestPolicy()
                val response = gatewayClient.request(request = initializationRequest, requestPolicy = requestPolicy, operationType = OperationType.INITIALIZATION)
                handleGatewayInitializationResponse(response.payload.initializationResponse)
            }
        } catch (e: Exception) {
            val failure = InitializationException.parseFrom(e)
            initializationFailure(startTime, failure)
            return@withContext
        }

        initializationSuccess(startTime)
    }

    private fun checkCanInitialize() {
        if (!sessionRepository.shouldInitialize) {
            throw InitializationException(message = MSG_GATEWAY_DENIED, reason = REASON_GATEWAY, reasonDebug = "!sessionRepository.shouldInitialize")
        }
    }

    private suspend fun initializationStart() {
        DeviceLog.debug("Unity Ads Initialization Start")
        sendDiagnosticEvent(event = INIT_STARTED)
        sessionRepository.initializationState = InitializationState.INITIALIZING
        eventObservers()
    }

    private suspend fun initializationSuccess(startTime: TimeMark) {
        DeviceLog.debug("Unity Ads Initialization Success")
        sendDiagnosticEvent(event = INIT_SUCCESS, value = startTime.elapsedMillis())
        storageManager.hasInitialized()
        initializeOM()
        clearCache()
        // We persist config and remove legacy config from disk so we can be ready for clean slate in case we switch back to non gateway flow.
        sessionRepository.persistNativeConfiguration()
        legacyConfigurationReader.currentConfiguration.deleteFromDisk()
        triggerInitializeListener.success()
        sessionRepository.initializationState = InitializationState.INITIALIZED
        sdkPropertiesManager.setInitialized(true)
        setupDiagnosticEvents()
    }

    private fun initializationFailure(startTime: TimeMark, e: InitializationException) {
        DeviceLog.debug("Unity Ads Initialization Failure: ${e.message}")
        sendDiagnosticEvent(event = INIT_FAILURE, value = startTime.elapsedMillis(), tags = getTags(e))
        triggerInitializeListener.error(INTERNAL_ERROR, e.message)
        sessionRepository.initializationState = InitializationState.FAILED
        sdkPropertiesManager.setInitialized(false)
        setupDiagnosticEvents()
    }

    private fun getTags(e: InitializationException): Map<String, String> {
        return buildMap {
            put(OPERATION, OperationType.INITIALIZATION.toString())
            put(REASON, e.reason)
            if (e.reasonDebug != null) {
                put(REASON_DEBUG, e.reasonDebug)
            }
        }
    }

    private fun setupDiagnosticEvents() {
        val config = sessionRepository.nativeConfiguration.diagnosticEvents
        diagnosticEventRepository.configure(config)
    }

    companion object {
        const val MSG_GATEWAY_DENIED = "Gateway communication failure"
        const val MSG_TIMEOUT = "Timeout"
        const val MSG_NETWORK = "Network"
        const val MSG_UNKNOWN = "Initialization failure"
    }
}
