package com.unity3d.ads.core.domain.events

import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.repository.TransactionEventRepository
import com.unity3d.ads.core.domain.GetRequestPolicy
import com.unity3d.ads.core.domain.GetUniversalRequestForPayLoad
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IAP_TRANSACTION
import com.unity3d.services.core.di.ServiceProvider.DEFAULT_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.NAMED_OTHER_REQ
import gateway.v1.UniversalRequestKt
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext




class TransactionEventObserver(
    private val getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
    
    private val defaultDispatcher: CoroutineDispatcher,
    private val transactionEventRepository: TransactionEventRepository,
    private val gatewayClient: GatewayClient,
    
    private val getRequestPolicy: GetRequestPolicy,
    
    private val iapTransactionStore: ByteStringDataSource,
    ) {
      private val isRunning = MutableStateFlow(false)

      suspend operator fun invoke() = withContext(defaultDispatcher) {
          // If we are already collecting events, do nothing
          if (isRunning.getAndUpdate { true }) return@withContext
          transactionEventRepository.transactionEvents.onEach { currentTransactionEventRequest ->
              val payload = UniversalRequestKt.payload {
                  transactionEventRequest = currentTransactionEventRequest
              }
              val fullRequest = getUniversalRequestForPayLoad(payload)

              val response = gatewayClient.request(
                  request = fullRequest,
                  requestPolicy = getRequestPolicy(),
                  operationType = OperationType.TRANSACTION_EVENT
              )

              if (response.error != null) {
                  // Set transaction timestamp value in the iapTransactionStore
                  // We can use the one stored in the request since it matches when it was created.
                  iapTransactionStore.set(fullRequest.sharedData.timestamps.timestamp.toByteString())
              }

          }.launchIn(CoroutineScope(defaultDispatcher))
      }
}