package com.unity3d.ads.core.domain.om

import android.content.Context
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.OMResult
import com.unity3d.ads.core.data.repository.OpenMeasurementRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OM_ACTIVATE_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OM_ACTIVATE_SUCCESS
import com.unity3d.ads.core.extensions.elapsedMillis
import org.koin.core.annotation.Single
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@Single
@OptIn(ExperimentalTime::class)
class InitializeOMAndroidSDK(
    private val context: Context,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sessionRepository: SessionRepository,
    private val omRepository: OpenMeasurementRepository
) : InitializeOMSDK {
    override suspend fun invoke() {
        if (sessionRepository.isOmEnabled) {
            val startTime = TimeSource.Monotonic.markNow()
            sendDiagnosticEvent(event = SendDiagnosticEvent.OM_ACTIVATE_STARTED)
            val result = omRepository.activateOM(context)
            when (result) {
                is OMResult.Success -> sendDiagnosticEvent(event = OM_ACTIVATE_SUCCESS, value = startTime.elapsedMillis())
                is OMResult.Failure -> {
                    sendDiagnosticEvent(
                        event = OM_ACTIVATE_FAILURE,
                        value = startTime.elapsedMillis(),
                        tags = buildMap {
                            put(SendDiagnosticEvent.REASON, result.reason)
                            result.reasonDebug?.let {
                                put(SendDiagnosticEvent.REASON_DEBUG, result.reasonDebug)
                            }
                        }
                    )
                }
            }
        }
    }
}