package com.unity3d.ads.core.domain.exposure

import android.util.Base64
import com.google.protobuf.ByteString
import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.adplayer.ExposedFunction
import com.unity3d.ads.adplayer.ExposedFunctionLocation
import com.unity3d.ads.core.data.model.AdData
import com.unity3d.ads.core.data.model.AdDataRefreshToken
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.ImpressionConfig
import com.unity3d.ads.core.data.repository.CampaignRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.CacheFile
import com.unity3d.ads.core.domain.GetAndroidAdPlayerContext
import com.unity3d.ads.core.domain.GetIsFileCache
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_DOWNLOAD_PRIORITY
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_DOWNLOAD_URL
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OMJS_SERVICE
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OMJS_SESSION
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_PARTNER
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_PARTNER_VERSION
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_VERSION
import com.unity3d.ads.core.domain.HandleOpenUrl
import com.unity3d.ads.core.domain.Refresh
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendPrivacyUpdateRequest
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.domain.om.AndroidOmStartSession
import com.unity3d.ads.core.domain.om.GetOmData
import com.unity3d.ads.core.domain.om.IsOMActivated
import com.unity3d.ads.core.domain.om.OmFinishSession
import com.unity3d.ads.core.domain.om.OmImpressionOccurred
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.utils.ContinuationFromCallback
import com.unity3d.services.UnityAdsConstants.Cache.CACHE_SCHEME
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_CACHE_DOMAIN
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_JS_URL_SERVICE
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_JS_URL_SESSION
import com.unity3d.services.core.api.Storage
import gateway.v1.OperativeEventRequestOuterClass
import gateway.v1.copy
import kotlinx.coroutines.flow.update
import org.json.JSONArray
import org.json.JSONObject
import org.koin.core.annotation.Named
import org.koin.core.annotation.Scope
import org.koin.core.annotation.Scoped
import kotlin.coroutines.suspendCoroutine

@Named(ExposedFunctionLocation.GET_AD_CONTEXT)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getAdContext(
    getAndroidAdPlayerContext: GetAndroidAdPlayerContext,
    adData: AdData,
    impressionConfig: ImpressionConfig,
    adDataRefreshToken: AdDataRefreshToken,
    isOMActivated: IsOMActivated,
    adObject: AdObject,
    ) = ExposedFunction {
    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA, adData.data)
        put(HandleInvocationsFromAdViewer.KEY_IMPRESSION_CONFIG, impressionConfig.data)
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN, adDataRefreshToken.data)
        put(HandleInvocationsFromAdViewer.KEY_NATIVE_CONTEXT, getAndroidAdPlayerContext())
        adObject.loadOptions.data?.let { loadOptions ->
            // Filter out adMarkup and objectId from loadOptions
            if (loadOptions.length() != 0) {
                put(HandleInvocationsFromAdViewer.KEY_LOAD_OPTIONS, loadOptions.keys().asSequence().fold(JSONObject()) { acc, key ->
                    if (key == "adMarkup" || key == "objectId") return@fold acc
                    acc.put(key, loadOptions[key])
                })
            }
        }
        if (isOMActivated()) {
            put(HandleInvocationsFromAdViewer.KEY_OMID, mapOf (
                KEY_OMJS_SESSION to OM_JS_URL_SESSION,
                KEY_OMJS_SERVICE to OM_JS_URL_SERVICE
            ))
        }
    }
}

@Named(ExposedFunctionLocation.GET_CONNECTION_TYPE)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getConnectionType(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.connectionType
}

@Named(ExposedFunctionLocation.GET_DEVICE_VOLUME)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getDeviceVolume(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.android.volume
}

@Named(ExposedFunctionLocation.GET_DEVICE_MAX_VOLUME)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getDeviceMaxVolume(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.android.maxVolume
}

@Named(ExposedFunctionLocation.GET_SCREEN_HEIGHT)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getScreenHeight(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.staticDeviceInfo().screenHeight
}

@Named(ExposedFunctionLocation.GET_SCREEN_WIDTH)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getScreenWidth(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.staticDeviceInfo().screenWidth
}

@Named(ExposedFunctionLocation.OPEN_URL)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun openUrl(handleOpenUrl: HandleOpenUrl) = ExposedFunction {
    val url = it[0] as String
    val params = it.getOrNull(1) as? JSONObject
    val packageName = params?.optString(HandleInvocationsFromAdViewer.KEY_PACKAGE_NAME)

    handleOpenUrl(url, packageName)
}

@Named(ExposedFunctionLocation.SEND_OPERATIVE_EVENT)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun sendOperativeEvent(getOperativeEventApi: GetOperativeEventApi, adObject: AdObject) = ExposedFunction {
    getOperativeEventApi(
        adObject = adObject,
        operativeEventType = OperativeEventRequestOuterClass.OperativeEventType.OPERATIVE_EVENT_TYPE_SPECIFIED_BY_AD_PLAYER,
        additionalEventData = Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

@Named(ExposedFunctionLocation.STORAGE_WRITE)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun writeStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.write(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }
}

@Named(ExposedFunctionLocation.STORAGE_READ)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun readStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.read(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }

}

@Named(ExposedFunctionLocation.STORAGE_DELETE)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun deleteStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.delete(
            args[0] as String,
            args[1] as String,
            ContinuationFromCallback(it)
        )
    }
}

@Named(ExposedFunctionLocation.STORAGE_CLEAR)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun clearStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.clear(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }
}

@Named(ExposedFunctionLocation.STORAGE_GET_KEYS)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getKeysStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.getKeys(
            args[0] as String,
            args[1] as String,
            args[2] as Boolean,
            ContinuationFromCallback(it)
        )
    }
}

@Named(ExposedFunctionLocation.STORAGE_GET)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.get(
            args[0] as String,
            args[1] as String,
            ContinuationFromCallback(it)
        )
    }
}


@Named(ExposedFunctionLocation.STORAGE_SET)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun setStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.set(
            args[0] as String,
            args[1] as String,
            args[2],
            ContinuationFromCallback(it)
        )
    }
}

@Named(ExposedFunctionLocation.GET_PRIVACY_FSM)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getPrivacyFsm(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.getPrivacyFsm().toBase64()
}

@Named(ExposedFunctionLocation.SET_PRIVACY_FSM)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun setPrivacyFsm(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.setPrivacyFsm(
        Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

@Named(ExposedFunctionLocation.GET_PRIVACY)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getPrivacy(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.getPrivacy().toBase64()
}

@Named(ExposedFunctionLocation.SET_PRIVACY)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun setPrivacy(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.setPrivacy(
        Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

@Named(ExposedFunctionLocation.GET_ALLOWED_PII)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getAllowedPii(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    Base64.encodeToString(deviceInfoRepository.allowedPii.value.toByteArray(), Base64.NO_WRAP)
}

@Named(ExposedFunctionLocation.SET_ALLOWED_PII)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun setAllowedPii(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    val allowedPiiUpdated = it[0] as JSONObject

    deviceInfoRepository.allowedPii.update { allowedPii ->
        allowedPii.copy {
            allowedPiiUpdated.optBoolean("idfa")?.let(::idfa::set)
            allowedPiiUpdated.optBoolean("idfv")?.let(::idfv::set)
        }
    }
}

@Named(ExposedFunctionLocation.GET_SESSION_TOKEN)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun getSessionToken(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.sessionToken.toBase64()
}

@Named(ExposedFunctionLocation.MARK_CAMPAIGN_STATE_SHOWN)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun markCampaignStateShown(campaignRepository: CampaignRepository, adObject: AdObject) = ExposedFunction {
    campaignRepository.setShowTimestamp(adObject.opportunityId)
}

@Named(ExposedFunctionLocation.REFRESH_AD_DATA)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun refreshAdData(refresh: Refresh, adObject: AdObject) = ExposedFunction {
    val refreshTokenByteString = if (it.isEmpty()) {
        ByteString.EMPTY
    } else {
        val refreshTokenJson = it[0] as JSONObject
        val refreshToken = refreshTokenJson.optString(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN)
        refreshToken.fromBase64()
    }

    val adRefreshResponse = refresh(refreshTokenByteString, adObject.opportunityId)

    if (adRefreshResponse.hasError()) throw IllegalArgumentException("Refresh failed")

    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA, adRefreshResponse.adData.toBase64())
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN, adRefreshResponse.adDataRefreshToken.toBase64())
        put(HandleInvocationsFromAdViewer.KEY_TRACKING_TOKEN, adRefreshResponse.trackingToken.toBase64())
    }
}

@Named(ExposedFunctionLocation.UPDATE_TRACKING_TOKEN)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun updateTrackingToken(adObject: AdObject) = ExposedFunction {
    val updateTrackingToken = it[0] as JSONObject
    val token = updateTrackingToken.optString("trackingToken")

    if (!token.isNullOrEmpty()) {
        adObject.trackingToken = token.fromBase64()
    }
}

@Named(ExposedFunctionLocation.SEND_PRIVACY_UPDATE_REQUEST)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun sendPrivacyUpdateRequest(sendPrivacyUpdateRequest: SendPrivacyUpdateRequest) = ExposedFunction {
    val privacyUpdateRequest = it[0] as JSONObject

    val privacyUpdateContentBase64 = privacyUpdateRequest.optString(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_CONTENT)
    val privacyUpdateVersion = privacyUpdateRequest.optInt(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_VERSION)

    val response = sendPrivacyUpdateRequest(privacyUpdateVersion, privacyUpdateContentBase64.fromBase64())

    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_VERSION, response.version)
        put(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_CONTENT, response.content.toBase64())
    }
}

@Named(ExposedFunctionLocation.SEND_DIAGNOSTIC_EVENT)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun sendDiagnosticEvent(sendDiagnosticEvent: SendDiagnosticEvent, adObject: AdObject) = ExposedFunction {
    val event = it[0] as String
    val tags = it[1] as JSONObject
    val tagsMap = buildMap {
        tags.keys().forEach { key ->
            put(key, tags.getString(key))
        }
    }
    val value = it.getOrNull(2)?.toString()?.toDouble()
    sendDiagnosticEvent(event = event, value = value, tags = tagsMap, adObject = adObject)
}

@Named(ExposedFunctionLocation.INCREMENT_BANNER_IMPRESSION_COUNT)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun incrementBannerImpressionCount(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.incrementBannerImpressionCount()
}

@Named(ExposedFunctionLocation.DOWNLOAD)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun download(cacheFile: CacheFile, adObject: AdObject) = ExposedFunction {
    val json = it[0] as JSONObject
    val url = json.getString(KEY_DOWNLOAD_URL)
    val headers = it.getOrNull(2) as JSONArray?
    val priority = json.optInt(KEY_DOWNLOAD_PRIORITY, 0)

    val result = cacheFile(url, adObject, headers, priority)

    when (result) {
        is CacheResult.Success -> "$CACHE_SCHEME://$AD_CACHE_DOMAIN/${result.cachedFile.name}.${result.cachedFile.extension}"
        is CacheResult.Failure -> url
    }
}

@Named(ExposedFunctionLocation.IS_FILE_CACHED)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun isFileCached(getIfFileCache: GetIsFileCache) = ExposedFunction {
    val fileUrl = it[0] as String
    getIfFileCache(fileUrl)
}

@Named(ExposedFunctionLocation.OM_START_SESSION)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun omStartSession(omStartSession: AndroidOmStartSession, adObject: AdObject) = ExposedFunction {
    val options = it[0] as JSONObject
    omStartSession(adObject, options)
}

@Named(ExposedFunctionLocation.OM_FINISH_SESSION)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun omFinishSession(omFinishSession: OmFinishSession, adObject: AdObject) = ExposedFunction {
    omFinishSession(adObject)
}

@Named(ExposedFunctionLocation.OM_IMPRESSION)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun omImpression(omImpressionOccurred: OmImpressionOccurred, adObject: AdObject) = ExposedFunction {
    val signalLoaded = it[0] as Boolean
    omImpressionOccurred(adObject, signalLoaded)
}

@Named(ExposedFunctionLocation.OM_GET_DATA)
@Scope(HandleInvocationsFromAdViewer::class)
@Scoped
internal fun omGetData(getOmData: GetOmData) = ExposedFunction {
    val data = getOmData()
    buildMap {
        put(KEY_OM_VERSION, data.version)
        put(KEY_OM_PARTNER, data.partnerName)
        put(KEY_OM_PARTNER_VERSION, data.partnerVersion)
    }
}