package com.unity3d.ads.core.domain

import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.repository.CacheRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.CACHE_SOURCE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_CACHE_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_CACHE_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.PROTOCOL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SIZE_KB
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.URL
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.toISO8859String
import org.json.JSONArray
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class CommonCacheFile(
    private val cacheRepository: CacheRepository,
    private val sendDiagnosticEvent: SendDiagnosticEvent
) : CacheFile {
    override suspend fun invoke(url: String, objectId: ByteString, headers: JSONArray?, priority: Int): CacheResult {
        val startTime = TimeSource.Monotonic.markNow()
        val result = cacheRepository.getFile(url, objectId.toISO8859String(), headers, priority)

        when(result) {
            is CacheResult.Success -> {
                sendDiagnosticEvent.invoke(
                    event = LOAD_CACHE_SUCCESS,
                    value = startTime.elapsedMillis(),
                    tags = mapOf(
                        CACHE_SOURCE to result.source.toString(),
                        URL to url,
                        PROTOCOL to result.cachedFile.protocol,
                    ),
                    intTags = mapOf(
                        SIZE_KB to (result.cachedFile.contentLength / 1024).toInt(),
                    )
                )
            }
            is CacheResult.Failure -> {
                sendDiagnosticEvent.invoke(
                    event = LOAD_CACHE_FAILURE,
                    value = startTime.elapsedMillis(),
                    tags = mapOf(
                        CACHE_SOURCE to result.source.toString(),
                        URL to url,
                        REASON to result.error.toString(),
                    ),
                )
            }
        }
        return result
    }
}