package com.unity3d.ads.core.data.repository

import android.content.Context
import com.unity3d.ads.core.data.datasource.CacheDataSource
import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.extensions.getDirectorySize
import com.unity3d.services.UnityAdsConstants.DefaultUrls.CACHE_DIR_NAME
import com.unity3d.services.core.extensions.memoize
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import okio.ByteString
import org.json.JSONArray
import java.io.File
import java.io.IOException
import java.util.concurrent.ConcurrentHashMap

class AndroidCacheRepository(
    private val ioDispatcher: CoroutineDispatcher,
    private val localCacheDataSource: CacheDataSource,
    private val remoteCacheDataSource: CacheDataSource,
    private val context: Context
) : CacheRepository {

    // cachedFiles fileName to CachedFile
    val cachedFiles = ConcurrentHashMap<String, CachedFile>()
    // neededFiles fileName to ObjectID <String, Map<ObjectIds>>
    val neededFiles = ConcurrentHashMap<String, MutableSet<String>>()
    private val cacheDir: File = initCacheDir()

    override suspend fun getFile(
        url: String,
        objectId: String,
        headers: JSONArray?,
        priority: Int
    ): CacheResult = withContext(ioDispatcher) {
        val filename = getHash(url)

        val localCachedFile = localCacheDataSource.getFile(cacheDir, filename, url, priority)
        if (localCachedFile is CacheResult.Success) {
            return@withContext localCachedFile.copy(cachedFile = localCachedFile.cachedFile.copy(objectId = objectId))
        }

        val newFile = File(getFilePath(filename))
        try {
            withContext(ioDispatcher) {
                newFile.createNewFile()
            }
        } catch (e: IOException) {
            return@withContext CacheResult.Failure(CacheError.FILE_IO_CREATE)
        }

        val fileResult = memoize(url) { remoteCacheDataSource.getFile(cacheDir, filename, url, priority) }
        if (fileResult is CacheResult.Success) {
            val cachedFile = fileResult.cachedFile.copy(objectId = objectId)
            addFileToCache(cachedFile)
        }

        return@withContext fileResult
    }

    override fun retrieveFile(fileName: String): CacheResult {
        val cachedFile = cachedFiles[fileName]
        return if (cachedFile != null) {
            CacheResult.Success(cachedFile, CacheSource.LOCAL)
        } else {
            CacheResult.Failure(CacheError.FILE_NOT_FOUND, CacheSource.LOCAL)
        }
    }

    override fun removeFile(cachedFile: CachedFile) = removeFileFromCache(cachedFile)

    override suspend fun doesFileExist(fileName: String): Boolean = cachedFiles.containsKey(fileName)

    fun getFilename(url: String): String = getHash(url)

    override suspend fun clearCache() {
        withContext(ioDispatcher) {
            cacheDir.listFiles()?.forEach { it.delete() }
        }
    }

    override suspend fun getCacheSize(): Long = withContext(ioDispatcher) {
        cacheDir.getDirectorySize()
    }

    private fun deleteFile(file: File?) {
        if (file != null && file.exists()) file.delete()
    }

    private fun getHash(path: String): String {
        val bytes = ByteString.of(*path.toByteArray())
        return bytes.sha256().hex()
    }

    private fun initCacheDir(): File {
        val dir = File(getCacheDirBase(), getCacheDirPath())
        dir.mkdirs()
        return dir
    }

    private fun getCacheDirBase(): File = context.cacheDir

    private fun getCacheDirPath(): String = CACHE_DIR_NAME

    private fun getFilePath(filename: String): String = cacheDir.absolutePath + File.separator + filename

    private fun addFileToCache(cachedFile: CachedFile) {
        cachedFiles[cachedFile.name] = cachedFile

        val observers = neededFiles[cachedFile.name] ?: mutableSetOf()
        observers.add(cachedFile.objectId)
        neededFiles[cachedFile.name] = observers
    }

    private fun removeFileFromCache(cachedFile: CachedFile) {
        cachedFiles.remove(cachedFile.name)
        neededFiles[cachedFile.name]?.remove(cachedFile.objectId)
        deleteFile(cachedFile.file)
    }
}