package com.twistpair.wave.thinclient.protocol.headers;

import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpCode;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpVersion;
import com.twistpair.wave.thinclient.util.CRC16Platform;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcpControlHeader extends IWtcpSubHeader
{
    public static final int SIZE                       = 8;
    public static final int SEQUENCE_NUMBER_BITS       = 16;

    public static final int TRANSACTION_ID_NUMBER_BITS = 16;
    public static final int TRANSACTION_ID_MAX         = (1 << TRANSACTION_ID_NUMBER_BITS) - 1;

    public static int getNextTransactionId(int lastTransactionId)
    {
        return (int) (++lastTransactionId % TRANSACTION_ID_MAX);
    }

    public int getOffset()
    {
        return WtcpHeader.SIZE;
    }

    public int getSize()
    {
        return SIZE;
    }

    public int getSequenceNumberBits()
    {
        return SEQUENCE_NUMBER_BITS;
    }

    public static final int OFFSET_START_CRC_CALCULATION = WtcpHeader.SIZE + 2;

    public int getMessageType()
    {
        return WtcpMessageType.Control;
    }

    public static class WtcpVersionedOpCode
    {
        public static final int CURRENT_VERSION = WtcpVersion.HeaderControl;

        /// <summary>
        /// Version = 0b11100000 0b00000000 (mask 0xE000)
        /// OpType  = 0b00011100 0b00000000 (mask 0x1C00)
        /// OpCode  = 0b00000011 0b11111111 (mask 0x03FF)
        /// </summary>
        private int             ushort0;                                    // TODO:(pv) Make this mutable WtcUInt16

        /// <summary>
        /// Version = 0b11100000 0b00000000 (mask 0xE000)
        /// </summary>
        public int getVersion()
        {
            return (((ushort0 & 0xE000) >> 13) + 1);
        }

        public void setVersion(int value)
        {
            int mask = 0xE000;
            ushort0 = ((ushort0 & ~mask) | (((value - 1) << 13) & mask));
        }

        /// <summary>
        /// OpType  = 0b00011100 0b00000000 (mask 0x1C00)
        /// </summary>
        public int getOpType()
        {
            return ((ushort0 & 0x1C00) >> 10);
        }

        public void setOpType(int value)
        {
            int mask = 0x1C00;
            ushort0 = ((ushort0 & ~mask) | (((byte) value << 10) & mask));
        }

        /// <summary>
        /// OpCode  = 0b00000011 0b11111111 (mask 0x03FF)
        /// </summary>
        public int getOpCode()
        {
            return ((ushort0 & 0x03FF) >> 0);
        }

        public void setOpCode(int value)
        {
            int mask = 0x03FF;
            ushort0 = ((ushort0 & ~mask) | ((value << 0) & mask));
        }

        /**
         * To be used only by WtcpControlHeader
         */
        protected WtcpVersionedOpCode() //
        {
            reset();
        }

        public WtcpVersionedOpCode(int opType, int opCode) //
        {
            this(CURRENT_VERSION, opType, opCode);
        }

        protected WtcpVersionedOpCode(int version, int opType, int opCode) //
        {
            //ushort0 = 0;
            setVersion(version);
            setOpType(opType);
            setOpCode(opCode);
        }

        public void dumpHostToNetworkOrder(IWtcMemoryStream buffer)
        {
            buffer.writeUInt16(ushort0);
        }

        public void loadNetworkToHostOrder(IWtcMemoryStream buffer)
        {
            ushort0 = buffer.readUInt16();
        }

        public void reset()
        {
            ushort0 = 0;
        }

        public String toString()
        {
            return toString('x');
        }

        public String toString(char format)
        {
            format = Character.toLowerCase(format);
            return new StringBuffer() //
            .append('{')
            //.append("ushort0=0b").append(WtcString.toBitString(ushort0, 16)).append(" (")
            .append("v=").append(WtcString.formatNumber(getVersion(), 1)).append(',') //
            .append("t=").append(WtcpOpType.toString(getOpType(), format)) //
            .append(",c=").append(WtcpOpCode.toString(getOpCode(), format)) //
            //.append(')') //
            .append('}') //
            .toString();
        }

        /*
        public static explicit operator ushort(WtcpVersionedOpCode opCode)
        {
            return opCode.ushort0;
        }

        public static explicit operator WtcpVersionedOpCode(ushort value)
        {
            return new WtcpVersionedOpCode(value);
        }
        */
    }

    public int                 crc;           // TODO:(pv) Make this mutable WtcUInt16
    public int                 sequenceNumber; // TODO:(pv) Make this mutable WtcUInt16
    public WtcpVersionedOpCode verOpCode;
    public int                 transactionId; // TODO:(pv) Make this mutable WtcUInt16

    public WtcpControlHeader() //
    {
        this(new WtcpVersionedOpCode());
    }

    public WtcpControlHeader(WtcpVersionedOpCode verOpCode) //
    {
        this(0, 0, verOpCode);
    }

    public WtcpControlHeader(int crc, int sequenceNumber, int opType, int opCode) //
    {
        this(crc, sequenceNumber, new WtcpVersionedOpCode(opType, opCode));
    }

    public WtcpControlHeader(int crc, int sequenceNumber, WtcpVersionedOpCode verOpCode) //
    {
        super();

        if (verOpCode == null)
        {
            throw new IllegalArgumentException("verOpCode cannot be null");
        }

        this.crc = crc;
        this.sequenceNumber = sequenceNumber;
        this.verOpCode = verOpCode;
        this.transactionId = 0;
    }

    public WtcpControlHeader(IWtcMemoryStream buffer) //
    {
        super(buffer);
    }

    public int calculateCrc(byte[] buffer, int length)
    {
        return CRC16Platform.update(buffer, OFFSET_START_CRC_CALCULATION, length - OFFSET_START_CRC_CALCULATION) ^ 0xFFFF;
    }

    public void dumpHostToNetworkOrder(IWtcMemoryStream buffer)
    {
        super.dumpHostToNetworkOrder(buffer);
        buffer.writeUInt16(crc);
        buffer.writeUInt16(sequenceNumber);
        verOpCode.dumpHostToNetworkOrder(buffer);
        buffer.writeUInt16(transactionId);
    }

    public boolean loadNetworkToHostOrder(IWtcMemoryStream buffer)
    {
        if (!super.loadNetworkToHostOrder(buffer))
        {
            return false;
        }
        crc = buffer.readUInt16();
        sequenceNumber = buffer.readUInt16();
        verOpCode.loadNetworkToHostOrder(buffer);
        transactionId = buffer.readUInt16();
        return true;
    }

    public void reset()
    {
        crc = sequenceNumber = transactionId = 0;

        // Semi-weak workaround for the super's constructor calling into this subclass' reset()
        // TODO:(pv) Consider improving this code a bit...
        if (verOpCode == null)
        {
            verOpCode = new WtcpVersionedOpCode();
        }
        else
        {
            verOpCode.reset();
        }
    }

    public String toString(char format)
    {
        format = Character.toLowerCase(format);
        StringBuffer sb = new StringBuffer() //
        .append('{');
        switch (format)
        {
            case 'd':
                sb.append("c=").append(WtcString.formatNumber(crc, 5)) //
                .append(",s=").append(WtcString.formatNumber(sequenceNumber, 5)) //
                .append(",o=").append(verOpCode.toString(format)) //
                .append(",t=").append(WtcString.formatNumber(transactionId, 5));
                break;
            default:
                sb.append("c=0x").append(WtcString.toHexString(crc, 2)) //
                .append(",s=0x").append(WtcString.toHexString(sequenceNumber, 2)) //
                .append(",o=").append(verOpCode.toString(format)) //
                .append(",t=0x").append(WtcString.toHexString(transactionId, 2));
                break;
        }
        return sb.append('}') //
        .toString();
    }

    public int getOpType()
    {
        return verOpCode.getOpType();
    }

    public int getOpCode()
    {
        return verOpCode.getOpCode();
    }

    public boolean isRequest()
    {
        //return (getOpType() & WtcpOpType.Request) == WtcpOpType.Request;
        return getOpType() == WtcpOpType.Request;
    }

    public boolean isResponse()
    {
        //return (getOpType() & WtcpOpType.Response) == WtcpOpType.Response;
        return getOpType() == WtcpOpType.Response;
    }

    public boolean isUnsolicited()
    {
        //return (getOpType() & WtcpOpType.Unsolicited) == WtcpOpType.Unsolicited;
        return getOpType() == WtcpOpType.Unsolicited;
    }

    public boolean isError()
    {
        //return (getOpType() & WtcpOpType.Error) == WtcpOpType.Error;
        return getOpType() == WtcpOpType.Error;
    }
}
