package com.twistpair.wave.thinclient.protocol.headers;

import com.twistpair.wave.thinclient.util.IWtcMemoryStream;

/// <summary>
/// An abstract class that:
/// 1) Defines properties common to all WTCP headers.
///     Offset, Size, and OffsetPayload
/// 2) Defines methods common to all WTCP headers.
///     Reset, ToString
/// 3) Defines the important DumpHostToNetworkOrder, and LoadNetworkToHostOrder serialization methods.
/// 
/// All subclasses must implement all of the above properties and methods.
/// </summary>
public abstract class IWtcpHeader
{
    /// <summary>
    /// Gets the offset (in bytes) that this header starts in the WtcpMessage.Buffer.
    /// </summary>
    public abstract int getOffset();

    /// <summary>
    /// Gets the packed size (in bytes) of all value types in this header.
    /// </summary>
    public abstract int getSize();

    /// <summary>
    /// Defines the bit size and thus the maximum sequence number the header supports.
    /// Used by NormalizeSequenceNumber to handle sequence number overflow.
    /// </summary>
    public abstract int getSequenceNumberBits();

    public int getSequenceNumberMaximum()
    {
        return ((1 << getSequenceNumberBits()) - 1);
    }

    /**
     * Wraps a sequence number if the given extendedSequenceNumber exceeds SequenceNumberMaximum.
     * @param extendedSequenceNumber
     * @return extendedSequenceNumber & SequenceNumberMaximum
     */
    public int normalizeSequenceNumber(long extendedSequenceNumber)
    {
        int sequenceNumberMaximum = getSequenceNumberMaximum();
        return (int) (extendedSequenceNumber & sequenceNumberMaximum);
    }

    /**
     * Gets the offset (in bytes) that a payload (if any) would start at.
     * This does *NOT* account for any additional sub-payload micro-offsets such as "Has16BitCrcAfterHeader".
     * @return getOffset() + getSize()
     */
    public int getPayloadOffset()
    {
        return getOffset() + getSize();
    }

    protected IWtcpHeader()
    {
        reset();
    }

    protected IWtcpHeader(IWtcMemoryStream buffer)
    {
        loadNetworkToHostOrder(buffer);
    }

    /// <summary>
    /// Serializes the header to the buffer in HostToNetworkOrder.
    /// All subclasses must call the base method first.
    /// If buffer.Length is less than this.OffsetPayload, then sets buffer.Length to this.OffsetPayload.
    /// If buffer.Length is greater than or equal to this.OffsetPayload, then leaves buffer.Length alone.
    /// This allows any content above this header segment to remain unaltered.
    /// Sets buffer.Position to this.Offset.
    /// </summary>
    /// <param name="buffer">The NetworkOrder buffer the dump/write to</param>
    public void dumpHostToNetworkOrder(IWtcMemoryStream buffer)
    {
        if (buffer.getLength() < getPayloadOffset())
        {
            buffer.setLength(getPayloadOffset());
        }
        buffer.setPosition(getOffset());
    }

    /// <summary>
    /// Serializes the header from the buffer in NetworkToHostOrder.
    /// All subclasses must call the base method first.
    /// If buffer.Length is less than this.OffsetPayload then immediately returns because there are not enough bytes to load from.
    /// </summary>
    /// <param name="buffer">The NetworkOrder buffer to load/read from</param>
    public boolean loadNetworkToHostOrder(IWtcMemoryStream buffer)
    {
        if (buffer.getLength() < getPayloadOffset())
        {
            return false;
        }
        buffer.setPosition(getOffset());
        return true;
    }

    /// <summary>
    /// Should set the packed value types in this header back to their default (optionally zero) values.
    /// </summary>
    public abstract void reset();

    /// <summary>
    /// Convert this Header to its equivalent [hex] String representation (for displaying/printing).
    /// </summary>
    /// <returns></returns>
    public String toString()
    {
        return toString('x');
    }

    /// <summary>
    /// Convert this Header to its equivalent String representation (for displaying/printing).
    /// </summary>
    /// <param name="format">char 'd' for decimal output, otherwise hex output.</param>
    /// <returns></returns>
    public abstract String toString(char format);
}
