package com.twistpair.wave.thinclient.protocol;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader.WtcpVersionedOpCode;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.util.WtcArrayQueue;

public class WtcpMessagePool
{
    private static final String TAG  = WtcLog.TAG(WtcpMessagePool.class);

    private final WtcArrayQueue pool = new WtcArrayQueue("QueueWtcpMessagePool");

    public void maintenance(boolean clear)
    {
        synchronized (pool)
        {
            pool.maintenance(clear);
        }
    }

    /**
     * Adds/Returns a WtcpMessage back to the message pool.
     * Care should be taken such that the pool does not grow unreasonably large.
     * If you only ever use the WtcpMessageCache to create messages then this shouldn't be a problem
     * (the private/protected constructor of WtcpMessage is intended to enforce this). 
     * @param message the a WtcpMessage to add to the pool
     */
    public void add(WtcpMessage message)
    {
        //WtcLog.error(TAG, "+add(" + message + ")");
        synchronized (pool)
        {
            pool.add(message);
        }
        //WtcLog.error(TAG, "-add(" + message + ")");
    }

    /**
     * Removes a WtcpMessage from the message pool and optionally resets the message.
     * @param reset true to reset the WtcpMessage pulled from the pool
     * @return a WtcpMessage pulled from the pool
     */
    protected WtcpMessage remove(boolean reset)
    {
        //WtcLog.error(TAG, "+remove(" + reset + ")");
        WtcpMessage message;
        synchronized (pool)
        {
            message = (WtcpMessage) pool.remove();
            if (reset)
            {
                message.reset();
            }
        }
        //WtcLog.error(TAG, "-remove(" + reset + ")");
        return message;
    }

    /**
     * Creates an empty WtcpMessage that is 100% blank (zeroed out).
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get()
    {
        //WtcLog.error(TAG, "+get()");
        WtcpMessage message;
        synchronized (pool)
        {
            if (pool.size() == 0)
            {
                //WtcLog.info(TAG, "Get: creating new WtcpMessage");
                message = new WtcpMessage();
            }
            else
            {
                //WtcLog.info(TAG, "Get: reusing pool WtcpMessage");
                message = remove(true);
            }
        }
        //WtcLog.error(TAG, "-get()");
        return message;
    }

    /**
     * Creates an empty WtcpMessage with the given WtcpHeader.
     * Moves the buffer pointer to the end of the header.
     * @param header
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(WtcpHeader header)
    {
        WtcpMessage message = get();
        message.setHeader(header);
        message.stream.setPosition(header.getSize());
        return message;
    }

    /**
     * Creates a WtcpMessage with the given WtcpMessageType.
     * @param messageType
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(byte messageType)
    {
        return get(new WtcpHeader(messageType, 0, 0));
    }

    /**
     * Creates a WtcpMessage with the given WtcpMediaHeader.
     * @param headerMedia
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(WtcpMediaHeader headerMedia)
    {
        WtcpMessage message = get();
        message.setSubHeader(headerMedia);
        return message;
    }

    /**
     * Creates a WtcpMessage Request with the given WtcpOpCode.
     * @param opCode
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(int opCode)
    {
        return get(opCode, null);
    }

    /**
     * Creates a WtcpMessage Request with the given WtcpOpCode and payload.
     * @param opCode
     * @param payload
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(int opCode, Object[] payload)
    {
        return get(WtcpOpType.Request, opCode, payload);
    }

    /**
     * Creates a WtcpMessage with the given WtcpOpType, WtcpOpCode.
     * @param opType
     * @param opCode
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(int opType, int opCode)
    {
        return get(opType, opCode, null);
    }

    /**
     * Creates a WtcpMessage with the given WtcpOpType, WtcpOpCode, and payload.
     * @param opType
     * @param opCode
     * @param payload
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(int opType, int opCode, Object[] payload)
    {
        return get(new WtcpControlHeader(new WtcpVersionedOpCode(opType, opCode)), payload);
    }

    /**
     * Creates a WtcpMessage with the given WtcpControlHeader.
     * @param controlHeader
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(WtcpControlHeader controlHeader)
    {
        return get(controlHeader, null);
    }

    /**
     * Creates a WtcpMessage with the given WtcpControlHeader and payload.
     * @param controlHeader
     * @param payload
     * @return a WtcpMessage pulled from the pool
     */
    public WtcpMessage get(WtcpControlHeader controlHeader, Object[] payload)
    {
        WtcpMessage message = get();
        message.setSubHeader(controlHeader);
        message.payloadAppend(payload);
        return message;
    }
}
