package com.twistpair.wave.thinclient.protocol;

import junit.framework.Assert;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.headers.IWtcpSubHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.protocol.types.IWtcpSendable;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcInt16;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcInt8;
import com.twistpair.wave.thinclient.util.WtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcScalar;
import com.twistpair.wave.thinclient.util.WtcString;
import com.twistpair.wave.thinclient.util.WtcUInt16;
import com.twistpair.wave.thinclient.util.WtcUInt32;
import com.twistpair.wave.thinclient.util.WtcUInt8;

public class WtcpMessage
{
    private static final String   TAG    = WtcLog.TAG(WtcpMessage.class);

    public final IWtcMemoryStream stream = new WtcMemoryStream();

    /**
     * Hide the public constructor.
     * Public WtcpMessages should only be created via WtcpMessageCache.removeOrCreate(...).
     * Creates a WtcpMessage with an unpopulated WtcpHeader with the buffer positioned to 0.
     */
    protected WtcpMessage()
    {
        reset();
    }

    public String toString()
    {
        return toString('a');
    }

    /**
     * Overload of the obligatory ToString representation of this class.
     * @param format 'a'=ASCII, 'd'=decimal, otherwise hex
     * @return a printable representation of this object.
     */
    public String toString(char format)
    {
        format = Character.toLowerCase(format);

        byte[] buffer = stream.getBuffer();
        int length = stream.getLength();
        int headerSize = header.getSize();

        StringBuffer sb = new StringBuffer();
        sb.append("{h");
        if (header == null)
        {
            sb.append("=NULL");
        }
        else
        {
            sb.append('(').append(headerSize).append(")=").append(header.toString(format));

            int payloadOffset = headerSize;
            int temp = stream.getPosition();
            switch (header.getMessageType())
            {
                case WtcpMessageType.Control:
                    WtcpControlHeader controlHeader = (WtcpControlHeader) getSubHeader();
                    sb.append(",c");
                    if (controlHeader == null)
                    {
                        sb.append("=NULL");
                    }
                    else
                    {
                        headerSize = controlHeader.getSize();
                        sb.append('(').append(headerSize).append(")=").append(controlHeader.toString(format));
                        payloadOffset += headerSize;
                    }
                    break;

                case WtcpMessageType.Media:
                    WtcpMediaHeader mediaHeader = (WtcpMediaHeader) getSubHeader();
                    sb.append(",m");
                    if (mediaHeader == null)
                    {
                        sb.append("=NULL");
                    }
                    else
                    {
                        headerSize = mediaHeader.getSize();
                        sb.append('(').append(headerSize).append(")=").append(mediaHeader.toString(format));
                        payloadOffset += headerSize;
                    }
                    // Cap media ToString output to at most 16 bytes...
                    length = Math.min((payloadOffset + 16), length);
                    break;
            }
            stream.setPosition(temp);
            int payloadLength = length - payloadOffset;

            // TODO:(pv) Proper parsing of payload bytes as potentially UTF8 encoded string
            sb.append(",p");
            if (payloadLength == 0)
            {
                sb.append("=NULL");
            }
            else
            {
                sb.append('(').append(payloadLength).append(")=");
                if (format != 'a' && format != 'd')
                {
                    sb.append(WtcString.toHexString(buffer, payloadOffset, payloadLength, true));
                }
                else
                {
                    for (int i = payloadOffset; i < length; i++)
                    {
                        switch (format)
                        {
                            case 'a':
                                byte b = buffer[i];
                                if (32 <= b && b < 127)
                                {
                                    if (b == 92)
                                    {
                                        sb.append("\\\\");
                                    }
                                    else
                                    {
                                        sb.append(WtcString.getString(buffer, i, 1));
                                    }
                                }
                                else
                                {
                                    sb.append('\\').append(b);
                                }
                                break;
                            case 'd':
                                if (i != payloadOffset)
                                {
                                    sb.append(' ');
                                }
                                sb.append(' ').append(WtcString.formatNumber(buffer[i], 3));
                                break;
                        }
                    }
                }
            }
        }

        sb.append('}');
        return sb.toString();
    }

    /**
     * Writes the Header to Buffer in Network order.
     */
    public void dumpHeaderHostToNetworkOrder()
    {
        header.dumpHostToNetworkOrder(stream);
    }

    public boolean getHasHeader()
    {
        return stream.getPosition() >= WtcpHeader.SIZE;
    }

    public boolean getHasSubHeader()
    {
        return getSubHeader() != null;
    }

    public byte getMessageType()
    {
        return header.getMessageType();
    }

    public void setMessageType(byte messageType)
    {
        header.setMessageType(messageType);
    }

    public boolean getIsMessageType(int messageType)
    {
        return getHasHeader() && getMessageType() == messageType;
    }

    /**
     * Only Control and Media messages should be encrypted.
     * @return true if the message should be encrypted, otherwise false.
     */
    public boolean getShouldBeCrypted()
    {
        return getIsMessageType(WtcpMessageType.Control) || getIsMessageType(WtcpMessageType.Media);
    }

    /**
     * Resets the Message for both Reading and Writing.
     * Sets WtcpMessage.Buffer.Length = WtcpHeader.SIZE, and implicitly initializes all values to zero.
     * @return WtcpHeader.SIZE
     */
    public int reset()
    {
        // TODO:(pv) reset control and media sub-headers?

        stream.setPosition(0);
        stream.setLength(0);
        stream.setLength(WtcpHeader.SIZE); // Auto zeros bytes from Position (0) to Length (WtcpHeader.SIZE)
        return stream.getLength();
    }

    // Make sure this is never set to null
    private final WtcpHeader header = new WtcpHeader();

    /**
     * Gets the WtcpHeader; will never be null.
     * Cannot be set to null; Trying to set to null invokes the default WtcpHeader() constructor.
     * Also sets the internal buffer length to zero; does NOT modify the buffer pointer.
     * @return the WtcpHeader
     */
    public WtcpHeader getHeader()
    {
        return header;
    }

    /**
     * Sets the WtcpHeader.
     * Cannot be set to null; Trying to set to null invokes the default WtcpHeader() constructor.
     * Also sets the internal buffer length to zero BUT DOES NOT MODIFY THE BUFFER POINTER!
     * @param value WtcpHeader for this message to use.
     */
    public void setHeader(WtcpHeader value)
    {
        if (value == null)
        {
            throw new IllegalArgumentException("value cannot be null");
        }
        stream.setLength(0);
        // TODO:(pv) reset() here?
        header.copyFrom(value);
        // TODO:(pv) reset() here?
        // TODO:(pv) buffer.setPosition(header.getSize()) here?
    }

    private final WtcpControlHeader subHeaderControl = new WtcpControlHeader();
    private final WtcpMediaHeader   subHeaderMedia   = new WtcpMediaHeader();

    /**
     * Loads the contents of any IWtcSubHeader from WtcpMessage.buffer from Network-to-Host byte order.
     * @return A reference to IWtcpSubHeader, if there is enough data in the buffer, otherwise null.
     */
    public IWtcpSubHeader getSubHeader()
    {
        IWtcpSubHeader subHeader = null;

        switch (getMessageType())
        {
            case WtcpMessageType.Control:
                subHeader = subHeaderControl;
                break;
            case WtcpMessageType.Media:
                subHeader = subHeaderMedia;
                break;
        }

        if (subHeader != null && stream.getLength() >= subHeader.getPayloadOffset())
        {
            subHeader.loadNetworkToHostOrder(stream);
            return subHeader;
        }

        return null;
    }

    /**
     * Writes the contents of the given IWtcSubHeader to WtcpMessage.buffer in Host-to-Network byte order.
     * Setting a subHeader to null is not supported; call "reset()" instead.
     * @param value non-null IWtcpSubHeader
     */
    public void setSubHeader(IWtcpSubHeader value)
    {
        if (value == null)
        {
            // TODO:(pv) Does it make sense to support setting the subHeader to null?
            // What affect would this have?
            throw new IllegalArgumentException("value cannot be null");
        }
        value.dumpHostToNetworkOrder(stream);
        Assert.assertEquals("buffer.getPosition() != value.getOffsetPayload()", stream.getPosition(), value.getPayloadOffset());
        header.setMessageType(value.getMessageType());
        header.setPayloadLength((int) (stream.getLength() - WtcpHeader.SIZE));
    }

    /**
     * *Copies* the message payload *to* the given buffer.
     * This function is intended to be used for debugging purposes only; calling this function may have undesirable [performance] side-effects.
     * @return
     */
    /*
    public byte[] payloadCopy()
    {
        byte[] buffer = stream.getBuffer();
        int payloadOffset = header.getPayloadOffset();
        int payloadLength = (int) (stream.getLength() - payloadOffset);
        byte[] payloadBuffer = new byte[payloadLength];
        System.arraycopy(buffer, payloadOffset, payloadBuffer, 0, payloadLength);
        return payloadBuffer;
    }
    */

    /*
    public void payloadSet(byte[] buffer)
    {
        payloadSet(buffer, 0, buffer.length);
    }
    */

    /**
     * Sets the payload from the beginning of the payload at the offset and length of the given buffer.
     * @param buffer
     * @param offset
     * @param count
     */
    public void payloadSet(byte[] buffer, int offset, int count)
    {
        IWtcpSubHeader subHeader = getSubHeader();
        stream.setPosition((subHeader == null) ? header.getPayloadOffset() : subHeader.getPayloadOffset());
        stream.write(buffer, offset, count);
    }

    public void payloadAppend(Object[] values)
    {
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                payloadAppend(values[i]);
            }
        }
    }

    /**
     * Smart but semi-heavy function; try not to call it too much if you can avoid it.
     * @param value
     */
    public void payloadAppend(Object value)
    {
        if (value == null)
        {
            throw new IllegalArgumentException("payloadAppend: null values are not allowed; try using a \"EMPTY\" property");
        }

        if (value instanceof IWtcpSendable)
        {
            ((IWtcpSendable) value).dumpHostToNetworkOrder(stream);
        }
        else if (value instanceof IWtcMemoryStream)
        {
            IWtcMemoryStream stream = ((IWtcMemoryStream) value);
            byte[] buffer = stream.getBuffer();
            int offset = stream.getPosition();
            int count = stream.getLength() - offset;
            this.stream.write(buffer, offset, count);
        }
        else if (value instanceof String)
        {
            stream.writeString((String) value);
        }
        else if (value instanceof WtcScalar)
        {
            if (value instanceof WtcInt8)
            {
                stream.writeInt8(((WtcInt8) value).value);
            }
            else if (value instanceof WtcUInt8)
            {
                stream.writeUInt8(((WtcUInt8) value).value);
            }
            else if (value instanceof WtcInt16)
            {
                stream.writeInt16(((WtcInt16) value).value);
            }
            else if (value instanceof WtcUInt16)
            {
                stream.writeUInt16(((WtcUInt16) value).value);
            }
            else if (value instanceof WtcInt32)
            {
                stream.writeInt32(((WtcInt32) value).value);
            }
            else if (value instanceof WtcUInt32)
            {
                stream.writeUInt32(((WtcUInt32) value).value);
            }
            else
            {
                throw new IllegalArgumentException("unhandled WtcScalar type");
            }
        }
        else
        {
            throw new IllegalArgumentException("unhandled type");
        }
    }

    /*
    public void payloadAppend(String value)
    {
        WtcpSerializer.dump(stream, value);
    }

    public void payloadAppendInt8(byte value)
    {
        WtcpSerializer.dumpHostToNetworkInt8(stream, value);
    }

    public void payloadAppendUInt8(short value)
    {
        WtcpSerializer.dumpHostToNetworkUInt8(stream, value);
    }

    public void payloadAppendInt16(short value)
    {
        WtcpSerializer.dumpHostToNetworkInt16(stream, value);
    }

    public void payloadAppendUInt16(int value)
    {
        WtcpSerializer.dumpHostToNetworkUInt16(stream, value);
    }

    public void payloadAppendInt32(int value)
    {
        WtcpSerializer.dumpHostToNetworkInt32(stream, value);
    }

    public void payloadAppendUInt32(long value)
    {
        WtcpSerializer.dumpHostToNetworkUInt32(stream, value);
    }

    public void payloadAppend(IWtcpSendable value)
    {
        value.dumpHostToNetworkOrder(stream);
    }

    public void payloadAppend(MemoryStream buffer)
    {
        payloadAppend(buffer.getBuffer());
    }

    public void payloadAppend(byte[] bytes)
    {
        payloadAppend(bytes, 0, bytes.length);
    }

    public void payloadAppend(byte[] bytes, int offset, int count)
    {
        stream.write(bytes, offset, count);
    }
    */
}
