package com.twistpair.wave.thinclient.net;

import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.regex.Pattern;

import com.twistpair.wave.thinclient.logging.WtcLog;

public class WtcNetworkExceptionPlatform extends IOException
{
    private static final String TAG = WtcLog.TAG(WtcNetworkExceptionPlatform.class);

    /**
     * Certain exceptions are ignorable and allow trying to connect to the next server
     * @param e
     * @return true if the Exception doesn't require an immediate stop
     */
    public static boolean mayTryNextServer(Exception e)
    {
        boolean mayTryNextServer = false;
        mayTryNextServer |= (e instanceof WtcNetworkUnknownHostException);
        mayTryNextServer |= (e instanceof SocketTimeoutException);
        // TODO:(pv) Add more exceptions as appropriate
        return mayTryNextServer;
    }

    protected static boolean isIOException(Throwable e)
    {
        return (e != null) && (e instanceof IOException);
    }

    protected static boolean isSocketException(Throwable e)
    {
        return (e != null) && (e instanceof SocketException);
    }

    /**
     * Cross-platform test for IOException related to DNS errors.
     * 
     * TODO:(pv) Similar code for other exceptions that could be encountered:
     * <ul>
     * <li>javax.microedition.io.ConnectionNotFoundException</li>
     * <li>instanceof IOException "APN is not specified"</li>
     * </ul>
     */
    public static boolean isUnknownHostException(Exception e)
    {
        if (!isIOException(e))
        {
            return false;
        }

        String className = e.getClass().getName().toLowerCase();

        boolean match = false;
        match |= className.equals("java.net.unknownhostexception");
        match |= className.equals("net.rim.device.cldc.io.dns.dnsexception");
        return match;
    }

    /**
     * "Hostname <X.X.X.X> was not verified"
     */
    public static boolean isHostnameNotVerified(Exception e)
    {
        if (!isIOException(e))
        {
            return false;
        }

        boolean match = false;

        String message = e.getMessage();
        if (message != null)
        {
            message = message.toLowerCase();
            match |= Pattern.matches("hostname <.*?> was not verified", message);
        }

        return match;
    }

    /**
     * "No route to host", "Network unreachable", or "Network is unreachable"
     */
    public static boolean isUnreachable(Exception e)
    {
        if (!isSocketException(e))
        {
            return false;
        }

        if (e instanceof ConnectException)
        {
            return true;
        }

        String message = e.getMessage();
        if (message != null)
        {
            message = message.toLowerCase();

            if (message.contains("no route to host"))
            {
                return true;
            }
            if (message.contains("network unreachable"))
            {
                return true;
            }
            if (message.contains("network is unreachable"))
            {
                return true;
            }
        }

        return false;
    }

    /**
     * Cross-platform test for IOException related to APN errors.
     * TODO:(pv) Find Android exception.
     */
    public static boolean isApnError(Throwable e)
    {
        if (!isIOException(e))
        {
            return false;
        }

        String message = e.getMessage();
        if (message != null)
        {
            message = message.toLowerCase();
            if (message.contains("apn is not specified"))
            {
                return true;
            }
        }

        return false;
    }

    public static boolean isRemoteConnectionClosed(SocketException e)
    {
        WtcLog.warn(TAG, "isRemoteConnectionClosed(...)");

        Throwable cause = e.getCause();
        if (cause != null)
        {
            WtcLog.info(TAG, "isRemoteConnectionClosed: cause=" + cause, cause);
            /*
            // cause may be instanceof [hidden/private] "libcore.io.ErrnoException"
            // TODO:(pv) Find way to get access to [hidden/private] "libcore.io.ErrnoException"
            //  Use reflection?
            ErrnoException errnoException = (ErrnoException) cause;
            if (errnoException.errno == 110)
            {
                return true;
            }
            */
        }

        String message = e.getMessage();
        if (message != null)
        {
            message = message.toLowerCase();
            if (message.contains("connection reset by peer"))
            {
                return true;
            }
        }

        return false;
    }

    public static boolean isLocalConnectionClosed(SocketException e)
    {
        WtcLog.warn(TAG, "isLocalConnectionClosed(...)");

        Throwable cause = e.getCause();
        if (cause != null)
        {
            WtcLog.info(TAG, "isLocalConnectionClosed: cause=" + cause, cause);
            // cause may be instanceof [hidden/private] "libcore.io.ErrnoException"
            /*
            // TODO:(pv) Find way to get access to [hidden/private] "libcore.io.ErrnoException"
            //  Use reflection?
            ErrnoException errnoException = (ErrnoException) cause;
            if (errnoException.errno == 110)
            {
                return true;
            }
            */
        }

        String message = e.getMessage();
        if (message != null)
        {
            message = message.toLowerCase();
            if (message.contains("connection timed out"))
            {
                return true;
            }
            if (message.contains("network is unreachable"))
            {
                return true;
            }
        }

        return false;
    }

    private static final long                 serialVersionUID = 786021788411188364L;

    public final WtcInetSocketAddressPlatform address;
    public final Exception                    innerException;

    protected WtcNetworkExceptionPlatform(WtcInetSocketAddressPlatform address, Exception innerException)
    {
        this.address = address;
        this.innerException = innerException;
    }

    public WtcNetworkExceptionPlatform(WtcUri uri, Exception innerException)
    {
        this(toWtcInetSocketAddress(uri), innerException);
    }

    public String toString()
    {
        return new StringBuffer() //
        .append('{') //
        .append("address=").append(address) //
        .append(", innerException=").append(innerException) //
        .append('}') //
        .toString();
    }

    // TODO:(pv) Move this to WtcNet, WtcUri, or WtcInetSocketAddress[Platform]
    protected static WtcInetSocketAddressPlatform toWtcInetSocketAddress(WtcUri uri)
    {
        int port = uri.getPort();
        if (port == -1)
        {
            if (WtcUri.URI_SCHEME_HTTP.equals(uri.getScheme()))
            {
                port = 80;
            }
            else if (WtcUri.URI_SCHEME_HTTPS.equals(uri.getScheme()))
            {
                port = 443;
            }
            else
            {
                port = 0;
            }
        }
        return new WtcInetSocketAddressPlatform(uri.getHost(), port);
    }

    /**
     * Cannot resolve a network address or uri.
     */
    public static class WtcNetworkUnknownHostException extends WtcNetworkExceptionPlatform
    {
        private static final long serialVersionUID = -4791427279849143246L;

        public WtcNetworkUnknownHostException(WtcInetSocketAddressPlatform address, Exception innerException)
        {
            super(address, innerException);
        }

        public WtcNetworkUnknownHostException(WtcUri uri, Exception innerException)
        {
            super(uri, innerException);
        }
    }

    /**
     * Cannot reach a network address or uri.
     */
    public static class WtcNetworkUnreachableException extends WtcNetworkExceptionPlatform
    {
        private static final long serialVersionUID = -5749677690136934889L;

        public WtcNetworkUnreachableException(WtcInetSocketAddressPlatform address, Exception innerException)
        {
            super(address, innerException);
        }

        public WtcNetworkUnreachableException(WtcUri uri, Exception innerException)
        {
            super(uri, translate(innerException));
        }

        public static Exception translate(Exception exception)
        {
            if (exception instanceof ConnectException)
            {
                /*
                // "wtcdev.twistpair.com/75.147.183.61:80 - Network is unreachable"
                String message = ((ConnectException) exception).getMessage();
                
                Pattern pattern = Pattern.compile("(.*?)  - Network is unreachable", Pattern.CASE_INSENSITIVE);
                Matcher matcher = pattern.matcher(message);
                if (matcher.)
                
                String message = ((ConnectException) exception).getMessage().toLowerCase();
                if (message.endsWith("network is unreachable"))
                {
                    pattern.

                }
                */
            }

            return exception;
        }
    }

    /**
     * Cannot verify the hostname of an SSL sertificate.
     * This is most likely due to the certificate being self-signing or not having an IP Address in it. 
     */
    public static class WtcNetworkHostnameNotVerifiedException extends WtcNetworkExceptionPlatform
    {
        private static final long serialVersionUID = -7278000041207334636L;

        public WtcNetworkHostnameNotVerifiedException(WtcUri uri, Exception innerException)
        {
            super(uri, innerException);
        }
    }

    /*
    public static class WtcNetworkApnException extends WtcNetworkException
    {
        // ?
    }
    */
}
