/**
 * @author pv@twistpair.com
 */

package com.twistpair.wave.thinclient;

import java.util.Calendar;
import java.util.Date;

import com.twistpair.wave.thinclient.WtcPingRequestRxTimeout.IPingRequestRxTimeoutListener;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackMessageReceiveTypeUnknownException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSecurityAgreementException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSessionCloseException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackUserProfilesEmptyException;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceMicrophone;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceMicrophone.IWtcMediaMicrophoneBufferListener;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceSpeaker;
import com.twistpair.wave.thinclient.media.WtcMediaExceptionPlatform;
import com.twistpair.wave.thinclient.net.WtcInetSocketAddressPlatform;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.net.WtcUriPlatform;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpEndpointFlags;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpErrorCodes;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpCode;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.WtcpMessage;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader.WtcpVersionedOpCode;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpAddressBookInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallAnswer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallDtmf;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallHangup;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallOffer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallProgress;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelActivity;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdErrorDictionary;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdList;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointProperties;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.protocol.types.WtcpKeyValueList;
import com.twistpair.wave.thinclient.protocol.types.WtcpProfileInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpStringList;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcInt16;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcInt8;
import com.twistpair.wave.thinclient.util.WtcSimpleDateFormatPlatform;
import com.twistpair.wave.thinclient.util.WtcString;
import com.twistpair.wave.thinclient.util.WtcVersionString;

/**
 * Stateless heavily async class that does the following:
 * <ul>
 * <li>Finds a Proxy given a http/https Locator address.</li>
 * <li>Connects to a Proxy given a wtcp address and starts separate TCP/UDP TX and Processing RX threads.</li>
 * <li>Secures the connection to/from the Proxy</li>
 * <li>Throws WtcStackUserProfilesEmpty if the User signing in has no Profiles.</li>
 * <li>Maintains a connection with the Proxy server by responding to Ping requests.</li>
 * <li>Maintains a connection with the Proxy server by sending Ping requests.</li>
 * <li>Tracks KEX and Control messages for timeouts.</li>
 * <li>Listens for IWtcMediaMicrophoneListener.onMicrophoneBuffer and sends media to Proxy.</li>
 * <li>Sends media received from Proxy to WtcSpeaker.</li>
 * <li>Reports progress/errors/etc to a WtcStackListener.</li>
 * </ul> 
 * 
 * In general, the difference between the WtcStack and the WtcClient, is that 
 * the stack does not care about the content of sessionid, profiles, channels, etc.
 * It will parse the values out, but again, it doesn't care about their content.
 * The content of these and similar values are higher level concepts handled by WtcClient.
 * 
 * TODO:(pv) Client actively regularly make Ping Requests to verify two-way health.
 * TODO:(pv) Make some/all of the thread daemon threads; setDaemon(true). Does this work on BB?
 * TODO:(pv) Consider passing a Runnable timeout to *ALL* stack requests?
 */
//
// BEGIN WtcStack
//
public class WtcStack //
                implements IWtcMediaMicrophoneBufferListener, //
                IPingRequestRxTimeoutListener
{
    private//
    static final//
    String               TAG    = WtcLog.TAG(//
                                WtcStack.class);

    private//
    static final boolean //
                         CENSOR = true;

    public static String censor(String s)
    {
        return (CENSOR) ? "*censored*" : s;
    }

    /**
     * Default ms value for the time it should take to establish a TCP connection to either the Locator or Proxy.
     */
    public//
    static final//
    int                               TIMEOUT_CONNECT_MS_DEFAULT                = 10 * 1000;   // 10 seconds

    /**
     * Default ms value for the time it should take to receive from the Proxy a Response to a message Request.<br>
     * ThreadConnection will dynamically calculate latency and adjust the actual message timeout accordingly.
     */
    public//
    static final//
    int                               TIMEOUT_REQUEST_TX_RESPONSE_RX_MS_DEFAULT = 30 * 1000;   // 30 seconds

    /**
     * Even if latency is calculated to be zero, always allow at least this much time for a response.
     */
    public//
    static final//
    int                               TIMEOUT_REQUEST_TX_RESPONSE_RX_MS_MIN     = 10 * 1000;   // 10 seconds

    public//
    static final//
    byte                              TIMEOUT_SESSION_SECONDS_MAX               = 120;

    public//
    static final//
    byte                              TIMEOUT_SESSION_SECONDS_MIN               = 5;

    /**
     * The "wtcp://" prefix can be used to bypass the locator and connect directly to a WTCP proxy.
     */
    public//
    static final//
    String                            URI_SCHEMA_WTCP                           = "wtcp";

    public//
    static final//
    int                               PORT_WTCP_DEFAULT                         = 4502;

    public//
    static final//
    String                            ID_ENDPOINT_SELF                          = "1";

    public//
    static final//
    int                               ID_CHANNEL_SPC                            = 0;

    // TODO:(pv) Change this to a property setter that starts/stops the UDP ThreadSend dynamically as appropriate.
    protected//
    final//
    boolean                          //
                                      useUdpForMedia                            = false;

    protected//
    final//
    WtcMediaDeviceMicrophone          microphone;

    protected//
    final//
    WtcMediaDeviceSpeaker             speaker;

    protected//
    final//
    WtcConnectionStatistics           connectionStatistics;

    protected//
    final//
    WtcVersionString                  version;

    protected//
    final//
    WtcUri[]                          remoteAddresses;

    /**
     * Some value of WtcKexPrimeSize. 
     */
    private//
    final//
    int                               kexSize;

    protected//
    final//
    int                               connectTimeoutMs;

    protected//
    final//
    WtcInetSocketAddressPlatform      localAddress;

    private//
    final//
    Object                            connectionSync                            = new Object();

    private WtcStackConnectionManager connectionManager;

    public WtcStack(//
    WtcMediaDeviceMicrophone microphone, WtcMediaDeviceSpeaker speaker, //
                    WtcConnectionStatistics connectionStatistics, //
                    WtcVersionString version, //                    
                    WtcUri[] remoteAddresses, //
                    int kexSize)
    {
        this(microphone, speaker, connectionStatistics, version, remoteAddresses, kexSize, TIMEOUT_CONNECT_MS_DEFAULT, null);
    }

    public WtcStack(//
    WtcMediaDeviceMicrophone microphone, WtcMediaDeviceSpeaker speaker, //
                    WtcConnectionStatistics connectionStatistics, //
                    WtcVersionString version, //
                    WtcUri[] remoteAddresses, //
                    int kexSize, //
                    int connectTimeoutMs)
    {
        this(microphone, speaker, connectionStatistics, version, remoteAddresses, kexSize, connectTimeoutMs, null);
    }

    public WtcStack(//
    WtcMediaDeviceMicrophone microphone, WtcMediaDeviceSpeaker speaker, //
                    WtcConnectionStatistics connectionStatistics, //
                    WtcVersionString version, //
                    WtcUri[] remoteAddresses, //
                    int kexSize, //
                    WtcInetSocketAddressPlatform localAddress)
    {
        this(microphone, speaker, connectionStatistics, version, remoteAddresses, kexSize, TIMEOUT_CONNECT_MS_DEFAULT,
                        localAddress);
    }

    public WtcStack( //
    WtcMediaDeviceMicrophone microphone, WtcMediaDeviceSpeaker speaker, //
                    WtcConnectionStatistics connectionStatistics, //
                    WtcVersionString version, //
                    WtcUri[] remoteAddresses, //
                    int kexSize, //
                    int connectTimeoutMs, //
                    WtcInetSocketAddressPlatform localAddress)
    {
        if (microphone == null)
        {
            throw new IllegalArgumentException("microphone cannot be null");
        }

        if (speaker == null)
        {
            throw new IllegalArgumentException("speaker cannot be null");
        }

        if (connectionStatistics == null)
        {
            throw new IllegalArgumentException("connectionStatistics cannot be null");
        }

        if (WtcUriPlatform.isNullOrEmpty(remoteAddresses))
        {
            throw new IllegalArgumentException("remoteAddresses cannot be null, empty, or contain null/EMPTY values");
        }

        this.microphone = microphone;
        this.speaker = speaker;
        this.connectionStatistics = connectionStatistics;
        this.version = version;
        this.remoteAddresses = remoteAddresses;
        this.kexSize = kexSize;
        this.connectTimeoutMs = connectTimeoutMs;
        this.localAddress = localAddress;
    }

    /**
     * <b>NOT</b> synchronized! (for performance reasons)<br>
     * All uses of this variable should save a reference and work off of that!
     */
    WtcStackListener _listener;

    public void setListener(WtcStackListener listener)
    {
        this._listener = listener;
    }

    /**
     * Silently ignores any Exception that would have been thrown (ex: IllegalThreadStateException on BB OS).
     * @param thread
     * @param join
     */
    protected static void interrupt(Thread thread, boolean join)
    {
        if (thread == null)
        {
            return;
        }

        String threadName = thread.getName();
        WtcLog.debug(TAG, "+" + threadName + ".interrupt()");
        try
        {
            thread.interrupt();
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, threadName + ".interrupt()", e);
        }
        WtcLog.debug(TAG, "-" + threadName + ".interrupt()");

        if (join)
        {
            WtcLog.debug(TAG, "+" + threadName + ".join()");
            try
            {
                thread.join();
            }
            catch (Exception e)
            {
                WtcLog.error(TAG, threadName + ".join()", e);
            }
            WtcLog.debug(TAG, "-" + threadName + ".join()");
        }
    }

    /**
     * @throws InterruptedException if interrupt() was called for this Thread
     */
    protected static void checkForInterruptedException() throws InterruptedException
    {
        Thread.sleep(0);
    }

    public void disconnect()
    {
        disconnect(null);
    }

    private void disconnect(Exception exitReason)
    {
        try
        {
            WtcLog.info(TAG, "+disconnect(" + WtcString.repr(exitReason) + ")");

            synchronized (connectionSync)
            {
                /*
                if (microphone != null)
                {
                    microphone.close(exitReason);
                }

                if (speaker != null)
                {
                    speaker.close(exitReason);
                }
                */

                if (connectionManager != null)
                {
                    connectionManager.disconnect(null);

                    // TODO:(pv) Wait for shutdown?
                    WtcStack.interrupt(connectionManager, false);

                    connectionManager = null;
                }
            }
        }
        finally
        {
            WtcLog.info(TAG, "-disconnect(" + WtcString.repr(exitReason) + ")");
        }
    }

    public void connect()
    {
        try
        {
            WtcLog.debug(TAG, "+connect()");

            synchronized (connectionSync)
            {
                disconnect();

                connectionManager = new WtcStackConnectionManager(this, 6, kexSize);
                //connectionManager.VerboseLog = verboseLog;
                //connectionManager.SetThreadPriority(networkThreadPriority, networkThreadBackground);
                connectionManager.start();
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-connect()");
        }
    }

    void processReceivedMessage(WtcpMessage message, boolean shouldIgnore) //
                    throws WtcStackMessageReceiveTypeUnknownException, WtcStackSecurityAgreementException, //
                    WtcStackSessionCloseException, WtcStackUserProfilesEmptyException//, //
    //WtcStackThreadProcessReceivedMessagesException
    {
        WtcpHeader header = message.getHeader();
        IWtcMemoryStream inputStream = message.stream;

        byte messageType = header.getMessageType();
        long length = inputStream.getLength();

        connectionStatistics.incRxed(messageType, length);

        if (shouldIgnore)
        {
            WtcLog.warn(TAG, "processReceivedMessage ignoring " + message);
            return;
        }

        switch (messageType)
        {
            case WtcpMessageType.Media:
                //connectionStatistics.incMessagesReceivedMedia();
                WtcpMediaHeader mediaHeader = (WtcpMediaHeader) message.getSubHeader();
                processMediaMessage(mediaHeader, inputStream);
                break;

            case WtcpMessageType.Control:
                //connectionStatistics.incMessagesReceivedControl();
                WtcpControlHeader controlHeader = (WtcpControlHeader) message.getSubHeader();
                processControlMessage(controlHeader, inputStream);
                break;

            case WtcpMessageType.UdpHello:
                processUdpHelloMessage(message);
                break;

            case WtcpMessageType.Hello:
                processHelloMessage(message);
                break;

            default:
                throw new WtcStackMessageReceiveTypeUnknownException(header.getMessageType());
        }
    }

    private void processHelloMessage(WtcpMessage message)
    {
        // HelloMessage not implemented
    }

    private void processUdpHelloMessage(WtcpMessage message)
    {
        // UdpHelloMessage not implemented
    }

    protected void processKeyExchangeMessage(IWtcMemoryStream inputStream) //
                    throws WtcStackSecurityAgreementException
    {
        try
        {
            WtcStackListener listener = this._listener;
            if (listener != null)
            {
                listener.onProxySecured(this, kexSize);
            }
        }
        catch (Exception e)
        {
            throw new WtcStackSecurityAgreementException(kexSize, e);
        }
    }

    private void processMediaMessage(WtcpMediaHeader mediaHeader, //
                    IWtcMemoryStream inputStream)
    {
        // TODO:(pv) Verify CRC, sequence #, etc...

        if (mediaHeader.getHas16BitCrcAfterHeader())
        {
            /*
            //Debugger.Break();
            int crc = WtcpSerializer.toUInt16(buffer, offset);
            offset += sizeof(ushort);
            count -= sizeof(ushort);

            // TODO:(pv) Verify CRC
            bool failed = false;
            if (failed)
            {
                Debugger.Break();
                // TODO:(pv) fire event, or just reject media?
                return;
            }
            */
            inputStream.incPosition(2); // 16 bits == 2 bytes
        }

        byte[] buffer = inputStream.getBuffer();
        int payloadOffset = inputStream.getPosition();
        int payloadLength = inputStream.getLength() - payloadOffset;

        try
        {
            // WtcMediaSpeaker.write(...) will add the buffer to a queue and and decode as necessary in a separate thread.
            speaker.write(buffer, payloadOffset, payloadLength);
        }
        catch (WtcMediaExceptionPlatform e)
        {
            WtcLog.error(TAG, "processMediaMessage: speaker.write(...)", e);
            // TODO:(pv) Should we abort the client at this point?
        }

        // The media buffer is most likely encoded (AMR, Speex, GSM, etc) and would not be directly usable to any listener here.
        // Thus, we do *NOT* event the *encoded* payload to the listener here.
        // If the app really wanted access to the payload the most usable place would be *after* WtcSpeaker decodes the buffer.  
        WtcStackListener listener = this._listener;
        if (listener != null)
        {
            listener.onMessageReceivedMedia(this, mediaHeader, payloadLength);
        }
    }

    public void onMicrophoneBuffer(byte[] buffer, int offset, int length)
    {
        sendMediaMessage(buffer, offset, length);
    }

    private final WtcpMediaHeader mAudioHeader = new WtcpMediaHeader(false);

    public void sendMediaMessage(byte[] buffer, int offset, int length)
    {
        //try
        //{
        // TODO:(pv) Does re-using the same private final audio header here cause any problems?
        WtcStackConnectionManager connectionManager = this.connectionManager;
        if (connectionManager != null)
        {
            WtcpMessage message = connectionManager.getMessage(mAudioHeader);//new WtcpMediaHeader(false));
            message.payloadSet(buffer, offset, length);
            connectionManager.send(message);
        }
        //}
        //catch (WtcStackException e)
        //{
        //    WtcLog.warn(TAG, "EXCEPTION onMicrophoneBuffer", e);
        //    // TODO:(pv) ignore? Or should we forcibly close the Microphone? 
        //}
    }

    private void logControlMessageIgnored(WtcpControlHeader controlHeader)
    {
        logWarnControlMessage(controlHeader, "Ignored");
    }

    private void LOG_CONTROL_MESSAGE_NOT_IMPLEMENTED(WtcpControlHeader controlHeader)
    {
        logWarnControlMessage(controlHeader, "Not Implemented");
    }

    private void logWarnControlMessage(WtcpControlHeader controlHeader, String text)
    {
        WtcLog.warn(TAG, "opType=" + WtcpOpType.toString(controlHeader.getOpType()) //
                        + ", opCode=\"" + WtcpOpCode.toString(controlHeader.getOpCode()) //
                        + "), transactionId=" + controlHeader.transactionId + ": " + text);
    }

    private void processControlMessage(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream) //
                    throws WtcStackSessionCloseException, WtcStackUserProfilesEmptyException//, //
    //WtcStackThreadProcessReceivedMessagesException
    {
        //int crc = controlHeader.crc;
        //int sequenceNumber = controlHeader.sequenceNumber;
        WtcpVersionedOpCode verOpCode = controlHeader.verOpCode;
        //int transactionId = controlHeader.transactionId;

        // TODO:(pv) Verify CRC, sequence #, etc...

        //int opType = verOpCode.getOpType();
        int opCode = verOpCode.getOpCode();

        if (controlHeader.isUnsolicited())
        {
            WtcLog.debug(TAG, "Unsolicited Message: " + WtcpOpCode.toString(opCode));
        }

        if (controlHeader.isError())
        {
            //Debugger.Break();
            WtcLog.error(TAG, "ERROR Message: " + WtcpOpCode.toString(opCode));
        }

        switch (opCode)
        {
        //
        // Misc internal WtcpOpCode messages; not usually directly evented to IWtcStackListener.
        //
            case WtcpOpCode.WTCPControlSignaling:
                WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                // TODO:(pv) How to properly bubble this up to the client/app and react?
                LOG_CONTROL_MESSAGE_NOT_IMPLEMENTED(controlHeader);
                break;
            case WtcpOpCode.Ping:
                processPing(controlHeader, inputStream);
                break;
            case WtcpOpCode.ReKex:
                // TODO:(pv) Handle unsolicited ReKex request from Proxy
                LOG_CONTROL_MESSAGE_NOT_IMPLEMENTED(controlHeader);
                break;

            //
            // Session
            //
            case WtcpOpCode.SessionOpen:
                processSessionOpen(controlHeader, inputStream);
                break;
            case WtcpOpCode.SessionClose:
                processSessionClose(controlHeader, inputStream);
                break;
            case WtcpOpCode.SetCredentials:
                processSetCredentials(controlHeader, inputStream);
                break;
            case WtcpOpCode.SessionResume:
                processSessionResume(controlHeader, inputStream);
                break;

            //
            // Channel
            //
            case WtcpOpCode.ChannelSetActive:
                processChannelSetActive(controlHeader, inputStream);
                break;
            case WtcpOpCode.ChannelList:
                LOG_CONTROL_MESSAGE_NOT_IMPLEMENTED(controlHeader);
                break;
            case WtcpOpCode.ChannelActivity:
                processChannelActivity(controlHeader, inputStream);
                break;
            case WtcpOpCode.ChannelPushToTalk:
                processChannelPushToTalk(controlHeader, inputStream);
                break;
            case WtcpOpCode.ChannelMute:
                processChannelMute(controlHeader, inputStream);
                break;
            case WtcpOpCode.ChannelPropertiesGet:
                processChannelPropertiesGet(controlHeader, inputStream);
                break;
            case WtcpOpCode.ChannelPropertiesSet:
                processChannelPropertiesSet(controlHeader, inputStream);
                break;

            //
            // Endpoint
            //
            case WtcpOpCode.EndpointPropertiesSet:
                processEndpointPropertiesSet(controlHeader, inputStream);
                break;
            case WtcpOpCode.EndpointLookup:
                processEndpointLookup(controlHeader, inputStream);
                break;
            case WtcpOpCode.EndpointPropertiesGet:
                processEndpointPropertiesGet(controlHeader, inputStream);
                break;
            case WtcpOpCode.EndpointPropertyFilterSet:
                processEndpointPropertyFilterSet(controlHeader, inputStream);
                break;

            //
            // Call
            //
            case WtcpOpCode.PhoneLinesSetActive:
                processPhoneLinesSetActive(controlHeader, inputStream);
                break;
            case WtcpOpCode.PhoneLineStatus:
                processPhoneLineStatus(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallMake:
                processCallMake(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallOffer:
                processCallOffer(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallAnswer:
                processCallAnswer(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallHangup:
                processCallHangup(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallProgress:
                processCallProgress(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallDtmf:
                processCallDtmf(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallPushToTalkOn:
                processCallPushToTalkOn(controlHeader, inputStream);
                break;
            case WtcpOpCode.CallPushToTalkOff:
                processCallPushToTalkOff(controlHeader, inputStream);
                break;

            //
            // Address Book
            //
            case WtcpOpCode.AddressBook:
                processAddressBook(controlHeader, inputStream);
                break;

            //
            // Unknown
            //
            default:
                // BUGBUG:(pv) if encryption fails [or does not match] erroneous opcodes will appear.
                // TODO:(pv) Stack should have checked CRC or done other verification for bad opcodes.
                logControlMessageIgnored(controlHeader);
                break;
        }

        // Unprocessed bytes occur when there is an unhandled message,
        // or if the protocol has changed to add more bytes to the payload end. 
        if (inputStream.getPosition() != inputStream.getLength())
        {
            int remain = inputStream.getLength() - inputStream.getPosition();
            WtcLog.warn(TAG, remain + " unprocessed/stray trailing bytes in payload!");
        }
    }

    /**
     * @param audioCodec
     * @param audioScale
     * @param sessionTimeoutSeconds
     * @param platformDescription
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendSessionOpen(WtcInt8 audioCodec, WtcInt8 audioScale, WtcInt8 sessionTimeoutSeconds,
                    String platformDescription)
    {
        try
        {
            WtcLog.info(TAG, "+sendSessionOpen(" + audioCodec + ", " + audioScale + ", " + sessionTimeoutSeconds + ", "
                            + WtcString.quote(platformDescription) + ")");

            if (sessionTimeoutSeconds.value < WtcStack.TIMEOUT_SESSION_SECONDS_MIN
                            || sessionTimeoutSeconds.value > WtcStack.TIMEOUT_SESSION_SECONDS_MAX)
            {
                throw new IllegalArgumentException("sessionTimeoutSeconds must be >= " + TIMEOUT_SESSION_SECONDS_MIN
                                + " and <= " + TIMEOUT_SESSION_SECONDS_MAX);
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.SessionOpen);
                message.payloadAppend(audioCodec);
                message.payloadAppend(audioScale);
                message.payloadAppend(sessionTimeoutSeconds);
                message.payloadAppend(platformDescription);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendSessionOpen(" + audioCodec + ", " + audioScale + ", " + sessionTimeoutSeconds + ", "
                            + WtcString.quote(platformDescription) + ")");
        }
    }

    private void processSessionOpen(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processSessionOpen");

            connectionManager.traceMessageRawBytes = connectionManager.traceMessageRawBytesAfterSessionOpen;

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                {
                    // TODO:(pv) Make this a first-class "WtcpSessionOpenResponse" class?
                    String sessionId = inputStream.readString();
                    WtcInt32 serverTime;
                    WtcVersionString serverVersion;
                    // serverTime was added in 5.2 Proxy; ignore it in responses from old Proxies
                    if (inputStream.getPosition() < inputStream.getLength())
                    {
                        serverTime = WtcInt32.valueOf(inputStream.readInt32(), false);
                    }
                    else
                    {
                        serverTime = null;
                    }
                    // serverVersion was added in 5.6 Proxy; ignore it in responses from old Proxies
                    if (inputStream.getPosition() < inputStream.getLength())
                    {
                        try
                        {
                            serverVersion = new WtcVersionString(inputStream.readString());
                        }
                        catch (Exception e)
                        {
                            serverVersion = null;
                        }
                    }
                    else
                    {
                        serverVersion = null;
                    }

                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onSessionOpen(this, controlHeader, sessionId, serverTime, serverVersion);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);

                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onSessionOpen(this, controlHeader, errorCode);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processSessionOpen");
        }
    }

    /**
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendSessionClose()
    {
        try
        {
            WtcLog.info(TAG, "+sendSessionClose()");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.SessionClose);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendSessionClose()");
        }
    }

    private void processSessionClose(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream) //
                    throws WtcStackSessionCloseException
    {
        try
        {
            WtcLog.debug(TAG, "+processSessionClose");

            WtcpErrorCode errorCode = null;

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Unsolicited:
                    errorCode = new WtcpErrorCode(inputStream);
                    break;
                case WtcpOpType.Response:
                    errorCode = null;
                    break;

                default:
                    //Debugger.Break();
                    break;
            }

            WtcStackListener listener = this._listener;
            if (listener != null)
            {
                listener.onSessionClose(this, controlHeader, errorCode);
            }

            // Receiver Thread catches this and exits gracefully:
            throw new WtcStackSessionCloseException(controlHeader.isUnsolicited(), errorCode);
        }
        finally
        {
            WtcLog.debug(TAG, "-processSessionClose");
        }
    }

    /**
     * @param username
     * @param password
     * @param license
     * @param profileId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendSetCredentials(String username, String password, String license, String profileId)
    {
        try
        {
            WtcLog.info(TAG,
                            "+sendSetCredentials(" + WtcString.quote(username) + ", " + censor(password) + ", "
                                            + WtcString.quote(license) + ", " + WtcString.quote(profileId) + ")");

            if (WtcString.isNullOrEmpty(profileId))
            {
                profileId = "";
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.SetCredentials);
                message.payloadAppend(username);
                message.payloadAppend(password);
                message.payloadAppend(license);
                message.payloadAppend(profileId);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.info(TAG,
                            "-sendSetCredentials(" + WtcString.quote(username) + ", " + censor(password) + ", "
                                            + WtcString.quote(license) + ", " + WtcString.quote(profileId) + ")");
        }
    }

    private void processSetCredentials(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream) //
                    throws WtcStackUserProfilesEmptyException
    {
        try
        {
            WtcLog.debug(TAG, "+processSetCredentials");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                {
                    String localEndpointId = inputStream.readString();
                    byte profileIndex = inputStream.readInt8();
                    WtcpProfileInfoList profiles = new WtcpProfileInfoList(inputStream);
                    WtcpChannelInfoList channels = new WtcpChannelInfoList(inputStream);
                    WtcpStringList phoneLines = new WtcpStringList(inputStream);

                    if (profiles.size() == 0)
                    {
                        throw new WtcStackUserProfilesEmptyException();
                    }

                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        WtcLog.info(TAG, "processSetCredentials(...): Received channels: " + channels);

                        listener.onSetCredentials(this, controlHeader, //
                                        localEndpointId, profileIndex, profiles, channels, phoneLines);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcpProfileInfoList profiles = null;
                    if (errorCode.getErrorCode() == WtcpErrorCodes.InvalidProfileId)
                    {
                        profiles = new WtcpProfileInfoList(inputStream);
                    }

                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onSetCredentials(this, controlHeader, errorCode, profiles);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processSetCredentials");
        }
    }

    // TODO:(pv) sendSessionResume(...)

    private void processSessionResume(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processSessionResume");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    String sessionId = inputStream.readString();
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onSessionResume(this, controlHeader, errorCode, sessionId);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processSessionResume");
        }
    }

    /**
     * @param channelIds
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelSetActive(WtcpChannelIdList channelIds)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelSetActive(" + channelIds + ")");

            if (channelIds == null)
            {
                channelIds = new WtcpChannelIdList();
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelSetActive);
                message.payloadAppend(channelIds);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelSetActive(" + channelIds + ")");
        }
    }

    private void processChannelSetActive(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelSetActive");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                    WtcpChannelIdErrorDictionary channelIdErrorDictionary = new WtcpChannelIdErrorDictionary(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelSetActive(this, controlHeader, channelIdErrorDictionary);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processChannelSetActive");
        }
    }

    /**
     * This function should rarely, if ever, be used.
     * It is implemented here for consistency, but it hasn't been tested and serves no real purpose.
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer sendChannelActivityRequest(WtcInt32 channelId)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelActivityRequest(" + channelId + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelActivity);
                message.payloadAppend(channelId);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelActivityRequest(" + channelId + ")");
        }
    }

    private void processChannelActivity(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelActivity");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                {
                    WtcpChannelActivity channelActivity = new WtcpChannelActivity(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelActivity(this, controlHeader, channelActivity);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelActivity(this, controlHeader, errorCode);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processChannelActivity");
        }
    }

    /**
     * @param channelIds *ABSOLUTE* list of channels to talk on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPushToTalk(WtcpChannelIdList channelIds)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelPushToTalk(" + channelIds + ")");

            if (channelIds == null)
            {
                channelIds = new WtcpChannelIdList();
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelPushToTalk);
                message.payloadAppend(channelIds);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelPushToTalk(" + channelIds + ")");
        }
    }

    private void processChannelPushToTalk(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelPushToTalk");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                    WtcpChannelIdErrorDictionary channelIdErrorDictionary = new WtcpChannelIdErrorDictionary(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelPushToTalk(this, controlHeader, channelIdErrorDictionary);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }

        }
        finally
        {
            WtcLog.debug(TAG, "-processChannelPushToTalk");
        }
    }

    /**
     * @param channelIds
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelMute(WtcpChannelIdList channelIds)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelMute(" + channelIds + ")");

            if (channelIds == null)
            {
                channelIds = new WtcpChannelIdList();
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelMute);
                message.payloadAppend(channelIds);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelMute(" + channelIds + ")");
        }
    }

    private void processChannelMute(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelMute");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                    WtcpChannelIdErrorDictionary channelIdErrorDictionary = new WtcpChannelIdErrorDictionary(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelMute(this, controlHeader, channelIdErrorDictionary);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }

        }
        finally
        {
            WtcLog.debug(TAG, "-processChannelMute");
        }
    }

    /**
     * @param channelId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesGet(int channelId, String[] propertyNames)
    {
        return sendChannelPropertiesGet(WtcInt32.valueOf(channelId, true), new WtcpStringList(propertyNames));
    }

    /**
     * @param channelId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesGet(WtcInt32 channelId, String[] propertyNames)
    {
        return sendChannelPropertiesGet(channelId, new WtcpStringList(propertyNames));
    }

    /**
     * @param channelId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesGet(int channelId, WtcpStringList propertyNames)
    {
        return sendChannelPropertiesGet(WtcInt32.valueOf(channelId, true), propertyNames);
    }

    /**
     * @param channelId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesGet(WtcInt32 channelId, WtcpStringList propertyNames)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelPropertiesGet(" + channelId + ", " + propertyNames + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelPropertiesGet);
                message.payloadAppend(channelId);
                message.payloadAppend(propertyNames);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelPropertiesGet(" + channelId + ", " + propertyNames + ")");
        }
    }

    private void processChannelPropertiesGet(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelPropertiesGet");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                {
                    int channelId = inputStream.readInt32();
                    WtcpKeyValueList properties = new WtcpKeyValueList(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelPropertiesGet(this, controlHeader, channelId, properties);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    int channelId = inputStream.readInt32();
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelPropertiesGet(this, controlHeader, channelId, errorCode);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        /*
        catch (Exception e)
        {
            WtcLog.error(TAG, "processChannelPropertiesGet", e);
            throw new WtcStackThreadProcessReceivedMessagesException("processChannelPropertiesGet", e);
        }
        */
        finally
        {
            WtcLog.debug(TAG, "-processChannelPropertiesGet");
        }
    }

    /**
     * @param channelId
     * @param properties
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesSet(int channelId, WtcpKeyValueList properties)
    {
        return sendChannelPropertiesSet(WtcInt32.valueOf(channelId, true), properties);
    }

    /**
     * @param channelId
     * @param properties
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendChannelPropertiesSet(WtcInt32 channelId, WtcpKeyValueList properties)
    {
        try
        {
            WtcLog.info(TAG, "+sendChannelPropertiesSet(" + channelId + ", " + properties + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.ChannelPropertiesSet);
                message.payloadAppend(channelId);
                message.payloadAppend(properties);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendChannelPropertiesSet(" + channelId + ", " + properties + ")");
        }
    }

    private void processChannelPropertiesSet(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processChannelPropertiesSet");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    int channelId = inputStream.readInt32();
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onChannelPropertiesSet(this, controlHeader, channelId, errorCode);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }

        }
        finally
        {
            WtcLog.debug(TAG, "-processChannelPropertiesSet");
        }
    }

    /**
     * @param properties
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertiesSet(WtcpKeyValueList properties)
    {
        return sendEndpointPropertiesSet(ID_ENDPOINT_SELF, properties);
    }

    /**
     * @param endpointId
     * @param properties
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertiesSet(String endpointId, WtcpKeyValueList properties)
    {
        try
        {
            WtcLog.info(TAG, "+sendEndpointPropertiesSet(" + WtcString.quote(endpointId) + ", " + properties + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.EndpointPropertiesSet);
                message.payloadAppend(endpointId);
                message.payloadAppend(properties);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendEndpointPropertiesSet(" + WtcString.quote(endpointId) + ", " + properties + ")");
        }
    }

    private void processEndpointPropertiesSet(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processEndpointPropertiesSet");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    if (errorCode.isError())
                    {
                        WtcStackListener listener = this._listener;
                        if (listener != null)
                        {
                            listener.onEndpointPropertiesSet(this, controlHeader, errorCode);
                        }
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processEndpointPropertiesSet");
        }
    }

    /**
     * Looks up a channel endpoint on the system wide "SPC" channel.
     * @param searchString name of endpoint to look up
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointLookup(String searchString)
    {
        final WtcInt32 channelId = WtcInt32.valueOf(ID_CHANNEL_SPC, true);
        return sendEndpointLookup(channelId, searchString);
    }

    /**
     * Looks up a channel endpoint on the given channel
     * @param channelId id of channel used to look up endpoint
     * @param searchString name of endpoint to look up
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointLookup(WtcInt32 channelId, String searchString)
    {
        final WtcInt32 flagsInclude = WtcInt32.valueOf(WtcpEndpointFlags.Visible, true);
        return sendEndpointLookup(channelId, flagsInclude, searchString);
    }

    /**
     * @param channelId id of channel used to look up endpoint
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param searchString name of endpoint to look up
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointLookup(WtcInt32 channelId, WtcInt32 flagsInclude, String searchString)
    {
        final WtcInt32 flagsExclude = WtcInt32.valueOf(WtcpEndpointFlags.None, true);
        return sendEndpointLookup(channelId, flagsInclude, flagsExclude, searchString);
    }

    /**
     * @param channelId id of channel used to look up endpoint
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param searchString name of endpoint to look up
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointLookup(WtcInt32 channelId, WtcInt32 flagsInclude, WtcInt32 flagsExclude, String searchString)
    {
        return sendEndpointLookup(channelId, flagsInclude, flagsExclude, WtcInt8.ZERO, WtcInt16.ZERO, searchString);
    }

    /**
     * @param channelId id of channel used to look up endpoint
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param pageSize number of endpoints to return per "page"; 0 to ignore
     * @param pageNumber page number of endpoints to return; 0 to ignore
     * @param searchString name of endpoint to look up
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointLookup(WtcInt32 channelId, WtcInt32 flagsInclude, WtcInt32 flagsExclude, //
                    WtcInt8 pageSize, WtcInt16 pageNumber, String searchString)
    {
        try
        {
            WtcLog.info(TAG, "+sendEndpointLookup(" + channelId + ", " + flagsInclude + ", " + flagsExclude //
                            + ", " + pageSize + ", " + pageNumber + ", " + WtcString.quote(searchString) + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.EndpointLookup);
                message.payloadAppend(channelId);
                message.payloadAppend(flagsInclude);
                message.payloadAppend(flagsExclude);
                message.payloadAppend(pageSize);
                message.payloadAppend(pageNumber);
                message.payloadAppend(searchString);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendEndpointLookup(" + channelId + ", " + flagsInclude + ", " + flagsExclude //
                            + ", " + pageSize + ", " + pageNumber + ", " + WtcString.quote(searchString) + ")");
        }
    }

    private void processEndpointLookup(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processEndpointLookup");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                {
                    int channelId = inputStream.readInt32();
                    short pageNumber = inputStream.readInt16();
                    short numberOfPages = inputStream.readInt16();
                    WtcpEndpointInfoList endpoints = new WtcpEndpointInfoList(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onEndpointLookup(this, controlHeader, channelId, pageNumber, numberOfPages, endpoints);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    int channelId = inputStream.readInt32();
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onEndpointLookup(this, controlHeader, channelId, errorCode);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processEndpointLookup");
        }
    }

    /**
     * @param endpointId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertiesGet(String endpointId, String[] propertyNames)
    {
        return sendEndpointPropertiesGet(endpointId, new WtcpStringList(propertyNames));
    }

    /**
     * @param endpointId
     * @param propertyNames
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertiesGet(String endpointId, WtcpStringList propertyNames)
    {
        try
        {
            WtcLog.info(TAG, "+sendEndpointPropertiesGet(" + WtcString.quote(endpointId) + ", " + propertyNames + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.EndpointPropertiesGet);
                message.payloadAppend(endpointId);
                message.payloadAppend(propertyNames);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendEndpointPropertiesGet(" + WtcString.quote(endpointId) + ", " + propertyNames + ")");
        }
    }

    private void processEndpointPropertiesGet(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processEndpointPropertiesGet");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                {
                    WtcpEndpointProperties endpointProperties = new WtcpEndpointProperties(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onEndpointPropertiesGet(this, controlHeader, endpointProperties);
                    }
                    break;
                }

                case WtcpOpType.Error:
                {
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onEndpointPropertiesGet(this, controlHeader, errorCode);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processEndpointPropertiesGet");
        }
    }

    /**
     * @param filterType
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertyFilterSet(int filterType, String filter)
    {
        return sendEndpointPropertyFilterSet(WtcInt32.valueOf(filterType, true), filter);
    }

    /**
     * @param filterType One of WtcpEndpointFilterType
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendEndpointPropertyFilterSet(WtcInt32 filterType, String filter)
    {
        try
        {
            WtcLog.info(TAG, "+sendEndpointPropertyFilterSet(" + filterType + ", " + WtcString.quote(filter) + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.EndpointPropertyFilterSet);
                message.payloadAppend(filterType);
                message.payloadAppend(filter);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendEndpointPropertyFilterSet(" + filterType + ", " + WtcString.quote(filter) + ")");
        }
    }

    private void processEndpointPropertyFilterSet(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processEndpointPropertyFilterSet");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    if (errorCode.isError())
                    {
                        WtcStackListener listener = this._listener;
                        if (listener != null)
                        {
                            listener.onEndpointPropertyFilterSet(this, controlHeader, errorCode);
                        }
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processEndpointPropertyFilterSet");
        }
    }

    private static final WtcSimpleDateFormatPlatform sLogCatDateFormat = new WtcSimpleDateFormatPlatform("MM-dd HH:mm:ss.SSS");

    //@Override
    public short onPingRequestRxTimeout(long timeoutMs, long elapsedMs, short lastPingRequestTxId)
    {
        Date ago = new Date(System.currentTimeMillis() - elapsedMs);
        Calendar cal = Calendar.getInstance();
        cal.setTime(ago);
        String since = sLogCatDateFormat.format(cal.getTime());
        WtcLog.warn(TAG, "$HEALTH: Did not RX PING Request in " + timeoutMs + "ms since " + since);
        WtcLog.warn(TAG, "$HEALTH: TXing PING Request; expecting RX PING Response or onMessageResponseTimeout.");

        short result = -1;

        WtcStackListener listener = this._listener;
        if (listener != null)
        {
            result = listener.onPingRequestRxTimeout(this, timeoutMs, elapsedMs, lastPingRequestTxId);
        }

        if (result == -1)
        {
            sendPingRequest(++lastPingRequestTxId);
        }

        return lastPingRequestTxId;
    }

    /**
     * @param id
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPingRequest(short id)
    {
        return sendPingRequest(new WtcInt16(id));
    }

    /**
     * @param id
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPingRequest(WtcInt16 id)
    {
        return sendPing(WtcpOpType.Request, id);
    }

    public void sendPingResponse(WtcInt16 id)
    {
        sendPing(WtcpOpType.Response, id);
    }

    private int          lastPingRequestId = 0;
    private final Object syncPingId        = new Object();

    /**
     * @param opType
     * @param id
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPing(int opType, WtcInt16 id)
    {
        try
        {
            WtcLog.info(TAG, "+sendPing(" + WtcpOpType.toString(opType) + ", " + id + ")");

            synchronized (syncPingId)
            {
                if (opType == WtcpOpType.Request)
                {
                    lastPingRequestId = id.value;
                }

                WtcStackConnectionManager connectionManager = this.connectionManager;
                if (connectionManager != null)
                {
                    WtcpMessage message = connectionManager.getMessage(opType, WtcpOpCode.Ping);
                    message.payloadAppend(id);
                    return connectionManager.send(message);
                }
                else
                {
                    return null;
                }
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendPing(" + WtcpOpType.toString(opType) + ", " + id + ")");
        }
    }

    private void processPing(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processPing");

            WtcInt16 pingId = null;

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Request:
                {
                    pingId = new WtcInt16(inputStream);
                    WtcLog.info(TAG, "lastPingRequestId=" + lastPingRequestId);
                    WtcLog.info(TAG, "RX Ping Request id=" + pingId + "; TX Ping Response id=" + pingId);
                    sendPingResponse(pingId);
                    break;
                }

                case WtcpOpType.Response:
                {
                    pingId = new WtcInt16(inputStream);
                    WtcLog.info(TAG, "RX Ping Response id=" + pingId);
                    break;
                }

                case WtcpOpType.Unsolicited:
                {
                    pingId = new WtcInt16(inputStream);
                    WtcLog.info(TAG, "RX Ping Unsolicited id=" + pingId + "; ignoring");
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }

            if (pingId != null)
            {
                WtcStackListener listener = this._listener;
                if (listener != null)
                {
                    listener.onPing(this, controlHeader, pingId.value);
                }
            }

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                connectionManager.maintenance();
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processPing");
        }
    }

    /**
     * @param phoneLine
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPhoneLineSetActive(String phoneLine)
    {
        WtcpStringList phoneLines = new WtcpStringList();
        phoneLines.addElement(phoneLine);
        return sendPhoneLinesSetActive(phoneLines);
    }

    /**
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPhoneLinesSetActive(String[] phoneLines)
    {
        return sendPhoneLinesSetActive(new WtcpStringList(phoneLines));
    }

    /**
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendPhoneLinesSetActive(WtcpStringList phoneLines)
    {
        try
        {
            WtcLog.info(TAG, "+sendPhoneLinesSetActive(" + phoneLines + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.PhoneLinesSetActive);
                message.payloadAppend(phoneLines);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendPhoneLinesSetActive(" + phoneLines + ")");
        }
    }

    private void processPhoneLinesSetActive(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processPhoneLinesSetActive");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onPhoneLinesSetActive(this, controlHeader, errorCode);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processPhoneLinesSetActive");
        }
    }

    private void processPhoneLineStatus(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processPhoneLineStatus");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Unsolicited:
                    String phoneLine = inputStream.readString();
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onPhoneLineStatus(this, controlHeader, phoneLine, errorCode);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processPhoneLineStatus");
        }
    }

    /**
     * @param callType
     * @param source
     * @param destination
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallMake(byte callType, String source, String destination)
    {
        return sendCallMake(new WtcInt8(callType), source, destination);
    }

    /**
     * @param callType One of WtcpCallType
     * @param source
     * @param destination
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallMake(WtcInt8 callType, String source, String destination)
    {
        try
        {
            WtcLog.info(TAG, "+sendCallMake(" + callType + ", " + WtcString.quote(source) + ", " + WtcString.quote(destination)
                            + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.CallMake);
                message.payloadAppend(callType);
                message.payloadAppend(source);
                message.payloadAppend(destination);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-sendCallMake(" + callType + ", " + WtcString.quote(source) + ", " + WtcString.quote(destination)
                            + ")");
        }
    }

    private void processCallMake(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallMake");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpCallInfo callInfo = new WtcpCallInfo(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallMake(this, controlHeader, callInfo);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallMake");
        }
    }

    private void processCallProgress(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallProgress");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Unsolicited:
                    WtcpCallProgress callProgress = new WtcpCallProgress(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallProgress(this, controlHeader, callProgress);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallProgress");
        }
    }

    /**
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallHangup(int callId)
    {
        return sendCallHangup(WtcInt32.valueOf(callId, true));
    }

    /**
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallHangup(WtcInt32 callId)
    {
        try
        {
            WtcLog.info(TAG, "+sendCallHangup(" + callId + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.CallHangup);
                message.payloadAppend(callId);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-sendCallHangup(" + callId + ")");
        }
    }

    private void processCallHangup(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallHangup");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                case WtcpOpType.Unsolicited:
                    WtcpCallHangup callHangup = new WtcpCallHangup(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallHangup(this, controlHeader, callHangup);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallHangup");
        }
    }

    private void processCallOffer(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallOffer");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Unsolicited:
                    WtcpCallOffer callOffer = new WtcpCallOffer(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallOffer(this, controlHeader, callOffer);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallOffer");
        }
    }

    /**
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallAnswer(int callId)
    {
        return sendCallAnswer(WtcInt32.valueOf(callId, true));
    }

    /**
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallAnswer(WtcInt32 callId)
    {
        try
        {
            WtcLog.info(TAG, "+sendCallAnswer(" + callId + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.CallAnswer);
                message.payloadAppend(callId);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-sendCallAnswer(" + callId + ")");
        }
    }

    private void processCallAnswer(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallAnswer");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Error:
                    WtcpCallAnswer callAnswer = new WtcpCallAnswer(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallAnswer(this, controlHeader, callAnswer);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallAnswer");
        }
    }

    /**
     * @param callId
     * @param digits
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallDtmf(int callId, String digits)
    {
        return sendCallDtmf(WtcInt32.valueOf(callId, true), digits);
    }

    /**
     * @param callId
     * @param digits
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallDtmf(WtcInt32 callId, String digits)
    {
        try
        {
            WtcLog.info(TAG, "+sendCallDtmf(" + callId + ", " + WtcString.quote(digits) + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                WtcpMessage message = connectionManager.getMessage(WtcpOpCode.CallDtmf);
                message.payloadAppend(callId);
                message.payloadAppend(digits);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-sendCallDtmf(" + callId + ", " + WtcString.quote(digits) + ")");
        }
    }

    private void processCallDtmf(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processCallDtmf");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                {
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallDtmf(this, controlHeader);
                    }
                    break;
                }
                case WtcpOpType.Error:
                {
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallDtmf(this, controlHeader, errorCode);
                    }
                    break;
                }
                case WtcpOpType.Unsolicited:
                {
                    WtcpCallDtmf callDtmf = new WtcpCallDtmf(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallDtmf(this, controlHeader, callDtmf);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processCallDtmf");
        }
    }

    /**
     * @param callId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallPushToTalk(int callId, boolean on)
    {
        return sendCallPushToTalk(WtcInt32.valueOf(callId, true), on);
    }

    /**
     * @param callId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendCallPushToTalk(WtcInt32 callId, boolean on)
    {
        try
        {
            WtcLog.info(TAG, "+sendCallPushToTalk(" + callId + ", " + on + ")");

            WtcStackConnectionManager connectionManager = this.connectionManager;
            if (connectionManager != null)
            {
                int opCode = (on) ? WtcpOpCode.CallPushToTalkOn : WtcpOpCode.CallPushToTalkOff;
                WtcpMessage message = connectionManager.getMessage(opCode);
                message.payloadAppend(callId);
                return connectionManager.send(message);
            }
            else
            {
                return null;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-sendCallPushToTalk(" + callId + ", " + on + ")");
        }
    }

    private void processCallPushToTalkOn(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processCallPushToTalkOn");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                case WtcpOpType.Error:
                    WtcInt32 callId = WtcInt32.valueOf(inputStream.readInt32(), true);
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallPushToTalkOn(this, controlHeader, callId, errorCode);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processCallPushToTalkOn");
        }
    }

    private void processCallPushToTalkOff(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.debug(TAG, "+processCallPushToTalkOff");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                case WtcpOpType.Unsolicited:
                case WtcpOpType.Error:
                    WtcInt32 callId = WtcInt32.valueOf(inputStream.readInt32(), true);
                    WtcpErrorCode errorCode = new WtcpErrorCode(inputStream);
                    WtcStackListener listener = this._listener;
                    if (listener != null)
                    {
                        listener.onCallPushToTalkOff(this, controlHeader, callId, errorCode);
                    }
                    break;

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-processCallPushToTalkOff");
        }
    }

    /**
     * @param version
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendAddressBookRequest(short version, String filter)
    {
        return sendAddressBookRequest(new WtcInt16(version), filter);
    }

    public static boolean FAKE_ADDRESS_BOOK = true;

    /**
     * @param version
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer sendAddressBookRequest(WtcInt16 version, String filter)
    {
        try
        {
            WtcLog.info(TAG, "+sendAddressBookRequest(" + version + ", " + WtcString.quote(filter) + ")");

            if (FAKE_ADDRESS_BOOK)
            {
                processAddressBook(new WtcpControlHeader(-1, -1, WtcpOpType.Response, WtcpOpCode.AddressBook), null);
                return null;
            }
            else
            {
                WtcStackConnectionManager connectionManager = this.connectionManager;
                if (connectionManager != null)
                {
                    WtcpMessage message = connectionManager.getMessage(WtcpOpCode.AddressBook);
                    message.payloadAppend(version);
                    message.payloadAppend(filter);
                    return connectionManager.send(message);
                }
                else
                {
                    return null;
                }
            }
        }
        finally
        {
            WtcLog.info(TAG, "-sendAddressBookRequest(" + version + ", " + WtcString.quote(filter) + ")");
        }
    }

    private void processAddressBook(WtcpControlHeader controlHeader, //
                    IWtcMemoryStream inputStream)
    {
        try
        {
            WtcLog.info(TAG, "+processAddressBook");

            switch (controlHeader.getOpType())
            {
                case WtcpOpType.Response:
                {
                    WtcInt16 version;
                    WtcpErrorCode errorCode;
                    WtcpAddressBookInfoList addressBookInfoList;

                    if (FAKE_ADDRESS_BOOK)
                    {
                        version = new WtcInt16(1);
                        errorCode = WtcpErrorCode.OK;
                        addressBookInfoList = WtcpAddressBookInfoList.FAKE;
                    }
                    else
                    {
                        version = new WtcInt16(inputStream);
                        errorCode = new WtcpErrorCode(inputStream);
                        addressBookInfoList = new WtcpAddressBookInfoList(inputStream);
                    }

                    WtcStackListener listener = _listener;
                    if (listener != null)
                    {
                        listener.onAddressBook(this, controlHeader, version, errorCode, addressBookInfoList);
                    }
                    break;
                }

                default:
                    //Debugger.Break();
                    break;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-processAddressBook");
        }
    }
}
//
// END WtcStack
//