package com.twistpair.wave.thinclient;

import java.util.Timer;
import java.util.TimerTask;

import com.twistpair.wave.thinclient.logging.WtcLog;

/*
 * Actively pings the Proxy if no Request has been RXed within the dynamically calculated session timeout interval.
 */
public class WtcPingRequestRxTimeout
{
    private static final String  TAG                  = WtcLog.TAG(WtcPingRequestRxTimeout.class);

    private static final boolean VERBOSE_LOG          = false;

    private final int            INTERVAL_PING_MS_MAX = WtcStack.TIMEOUT_SESSION_SECONDS_MAX * 1000;
    private final int            INTERVAL_PING_MS_MIN = WtcStack.TIMEOUT_SESSION_SECONDS_MIN * 1000;

    public interface IPingRequestRxTimeoutListener
    {
        /**
         * @param timeoutMs timeout requested
         * @param elapsedMs timeout actual
         * @param lastPingRequestTxId the previously Transmitted Ping Request PingId
         * @return the next Transmitted Ping Request PingId, or -1 if not TXed
         */
        short onPingRequestRxTimeout(long timeoutMs, long elapsedMs, short lastPingRequestTxId);
    }

    private final IPingRequestRxTimeoutListener listener;
    private final WtcConnectionStatistics       connectionStatistics;
    private final Object                        syncCancel = new Object();

    private boolean                             closed;
    private Timer                               timer;
    private TimerTask                           timerTask;

    private short                               lastPingRequestTxId;

    private long                                timePingInterval;
    private long                                timeStarted;

    public WtcPingRequestRxTimeout(IPingRequestRxTimeoutListener listener, WtcConnectionStatistics connectionStatistics)
    {
        this.listener = listener;
        this.connectionStatistics = connectionStatistics;

        timer = new Timer();

        lastPingRequestTxId = 0;

        timePingInterval = INTERVAL_PING_MS_MAX;
    }

    public void close()
    {
        try
        {
            WtcLog.debug(TAG, "+close()");

            synchronized (syncCancel)
            {
                cancel();

                closed = true;

                if (timer != null)
                {
                    WtcLog.debug(TAG, "timer.cancel()");
                    timer.cancel();
                    timer = null;
                }
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-close()");
        }
    }

    public long cancel()
    {
        //try
        //{
        //    WtcLog.debug(TAG, "+cancel()");

        synchronized (syncCancel)
        {
            if (closed)
            {
                return -1;
            }

            long elapsedMs;

            if (timer != null && timerTask != null)
            {
                //WtcLog.debug(TAG, "timerTask.cancel()");
                timerTask.cancel();
                timerTask = null;

                long now = System.currentTimeMillis();
                elapsedMs = now - timeStarted;
                //WtcLog.debug(TAG, "now=" + now + ", timeStarted=" + timeStarted + ": elapsedMs=" + elapsedMs);
            }
            else
            {
                //WtcLog.debug(TAG, "timer == null; defaulting to INTERVAL_PING_MS_MAX");
                elapsedMs = INTERVAL_PING_MS_MAX;
            }

            //WtcLog.debug(TAG, "cancel(): elapsedMs=" + elapsedMs);
            return elapsedMs;
        }
        //}
        //finally
        //{
        //    WtcLog.debug(TAG, "-cancel()");
        //}
    }

    public void reset(boolean calculatePingInterval)
    {
        //try
        //{
        //    WtcLog.debug(TAG, "+reset(calculatePingInterval=" + calculatePingInterval + ")");

        synchronized (syncCancel)
        {
            long elapsedMs = cancel();
            if (elapsedMs < 0)
            {
                return;
            }

            if (calculatePingInterval)
            {
                long now = System.currentTimeMillis();
                timePingInterval = now - timeStarted;
                //WtcLog.debug(TAG, "now=" + now + ", timeStarted=" + timeStarted + ": timePingInterval=" + timePingInterval);

                if (VERBOSE_LOG)
                {
                    WtcLog.error(TAG, "$HEALTH: RXed PING Request in " + timePingInterval + "ms");
                }
            }

            long requestTimeoutMs = connectionStatistics.latency.getRequestTimeoutMs();
            int timeoutMs =
                (int) (Math.max(INTERVAL_PING_MS_MIN, Math.min(timePingInterval, INTERVAL_PING_MS_MAX)) + requestTimeoutMs);
            //WtcLog.debug(TAG, "requestTimeoutMs=" + requestTimeoutMs + ", timePingInterval=" + timePingInterval + ": timeoutMs=" + timeoutMs);

            schedule(timeoutMs);
        }
        //}
        //finally
        //{
        //    WtcLog.debug(TAG, "-reset(calculatePingInterval=" + calculatePingInterval + ")");
        //}
    }

    protected void schedule(final int timeoutMs)
    {
        //try
        //{
        //    WtcLog.debug(TAG, "+schedule(" + timeoutMs + ")");

        synchronized (syncCancel)
        {
            long elapsedMs = cancel();
            if (elapsedMs < 0)
            {
                return;
            }

            timerTask = new TimerTask()
            {
                public void run()
                {
                    WtcPingRequestRxTimeout.this.run(timeoutMs);
                }
            };

            if (VERBOSE_LOG)
            {
                WtcLog.error(TAG, "$HEALTH: Waiting up to " + timeoutMs + "ms for next RX PING Request");
            }

            this.timeStarted = System.currentTimeMillis();
            timer.schedule(timerTask, timeoutMs);
            //WtcLog.debug(TAG, "schedule: scheduled task; timeStart=" + timeStart + ", timeoutMs=" + timeoutMs);
        }
        //}
        //finally
        //{
        //    WtcLog.debug(TAG, "-schedule(" + timeoutMs + ")");
        //}
    }

    protected void run(int timeoutMs)
    {
        try
        {
            WtcLog.warn(TAG, "+run(timeoutMs=" + timeoutMs + ")");

            // Lock here to prevent event from firing during another thread's call to cancel()
            synchronized (WtcPingRequestRxTimeout.this)
            {
                if (timerTask == null)
                {
                    return;
                }

                long elapsedMs = System.currentTimeMillis() - timeStarted;

                lastPingRequestTxId = listener.onPingRequestRxTimeout(timeoutMs, elapsedMs, lastPingRequestTxId);

                cancel();
            }
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, "EXCEPTION run(timeoutMs=" + timeoutMs + ")", e);
        }
        finally
        {
            WtcLog.warn(TAG, "-run(timeoutMs=" + timeoutMs + ")");
        }
    }
}
