package com.twistpair.wave.thinclient;

import java.io.IOException;

import org.w3c.dom.Element;

import com.twistpair.wave.thinclient.WtcLocatorException.WtcLocatorResponseInvalidException;
import com.twistpair.wave.thinclient.kexcrypto.WtcCryptoUtilPlatform;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.net.WtcNet;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.net.WtcUriPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcLocator
{
    private static final String TAG                       = WtcLog.TAG(WtcLocator.class);

    private static final String DEFAULT_WAVE_LOCATOR_PATH = "/wave/_interface/proxy_locator.asp";

    /**
     * 
     * @param uriLocator
     * @return WtcLocatorResponse
     * @throws IOException
     * @throws WtcLocatorResponseInvalidException
     */
    public static WtcLocatorResponse locateProxies(WtcUri uriLocator) //
                    throws IOException, WtcLocatorResponseInvalidException
    {
        try
        {
            WtcLog.info(TAG, "+locateProxies(" + uriLocator + ")");

            Element xmlDoc = WtcNet.makeRequestXml(uriLocator);
            return new WtcLocatorResponse(xmlDoc);
        }
        finally
        {
            WtcLog.info(TAG, "-locateProxies(" + uriLocator + ")");
        }
    }

    /**
     * Utility method for createLocatorUri
     * @param uri
     * @param key
     * @param defaultValue
     * @return WtcUri
     */
    public static WtcUri verifyQuery(WtcUri uri, String key, long defaultValue)
    {
        return verifyQuery(uri, key, String.valueOf(defaultValue));
    }

    /**
     * Utility method for createLocatorUri
     * @param uri
     * @param key
     * @param defaultValue
     * @return WtcUri
     */
    public static WtcUri verifyQuery(WtcUri uri, String key, String defaultValue)
    {
        String queryParameter = uri.getQueryParameter(key);
        if (WtcString.isNullOrEmpty(queryParameter))
        {
            WtcUri.Builder builder = uri.buildUpon();
            builder.appendQueryParameter(key, defaultValue);
            uri = builder.build();
        }
        return uri;
    }

    /**
     * Creates a WtcUriPlatform that can be used to connect to a WTC Locator.
     * Appends a unique nonce to the end of the query to always bypass any system cache.
     * @param uriString example: "https://wtcdev.twistpair.com" or "wtcdev.twistpair.com" (implies "http://")
     * @return WtcUri
     */
    public static WtcUri createLocatorUri(String uriString)
    {
        return createLocatorUri(uriString, null, null);
    }

    /**
     * @param uriString
     * @param ssl
     * @return WtcUri
     */
    public static WtcUri createLocatorUri(String uriString, boolean ssl)
    {
        return createLocatorUri(prependRequestProtocol(uriString, ssl), null, null);
    }

    /**
     * @param uriString
     * @param ssl
     * @param packageCode
     * @param appVersion
     * @return WtcUri
     */
    public static WtcUri createLocatorUri(String uriString, boolean ssl, String packageCode, String appVersion)
    {
        return createLocatorUri(prependRequestProtocol(uriString, ssl), packageCode, appVersion);
    }

    /**
     * Creates a WtcUriPlatform that can be used to connect to a WTC Locator.
     * Appends a unique nonce to the end of the query to always bypass any system cache.
     * @param uriString example: "https://wtcdev.twistpair.com" or "wtcdev.twistpair.com" (implies "http://")
     * @param packageCode Optional package code to pass to the Locator as a "pc=" parameter; Use null/"" to not set.  
     * @param appVersion Optional application version to pass to the Locator as a "ver=" parameter; Use null/"" to not set. 
     * @return WtcUri
     */
    public static WtcUri createLocatorUri(String uriString, String packageCode, String appVersion)
    {
        if (WtcString.isNullOrEmpty(uriString))
        {
            throw new IllegalArgumentException("uriString cannot be null or empty");
        }

        WtcUri uriLocator = WtcUriPlatform.parse(uriString);

        String host = uriLocator.getHost();
        if (WtcString.isNullOrEmpty(host))
        {
            throw new IllegalArgumentException("uriString must contain a host");
        }

        String path = uriLocator.getPath();
        if (WtcString.isNullOrEmpty(path))
        {
            WtcUri.Builder builder = uriLocator.buildUpon();
            builder.path(DEFAULT_WAVE_LOCATOR_PATH);
            uriLocator = builder.build();
        }

        uriLocator = verifyQuery(uriLocator, "action", "getproxy");
        if (!WtcString.isNullOrEmpty(packageCode))
        {
            uriLocator = verifyQuery(uriLocator, "pc", packageCode);
        }
        if (!WtcString.isNullOrEmpty(appVersion))
        {
            uriLocator = verifyQuery(uriLocator, "ver", appVersion);
        }

        // Add a unique nonce so that we always bypass any system cache
        uriLocator = verifyQuery(uriLocator, "nonce", WtcCryptoUtilPlatform.getRandomInt64());

        // TODO:(pv) Append any OS specific network options [hidden in addressLocal?]
        // Example: BB deviceside=true, wifi, apn, etc...
        //urlLocator += networkOptions;

        // TODO:(pv) Bind request to a specified NIC.
        // See http://blogs.msdn.com/malarch/archive/2005/09/13/466664.aspx
        //request.ServicePoint.BindIPEndPointDelegate = delegate(ServicePoint servicePoint, IPEndPoint remoteEndPoint, int retryCount)
        //{
        //    return new IPEndPoint(LocalAddress, 0);
        //};

        return uriLocator;
    }

    private static String prependRequestProtocol(String uriString, boolean ssl)
    {
        if (WtcString.isNullOrEmpty(uriString))
        {
            throw new IllegalArgumentException("uriString cannot be null or empty");
        }

        if (!uriString.startsWith(WtcStack.URI_SCHEMA_WTCP))
        {
            if (ssl && !uriString.startsWith(WtcUri.URI_SCHEME_HTTPS))
            {
                if (uriString.startsWith(WtcUri.URI_SCHEME_HTTP + "://"))
                {
                    uriString = WtcString.replaceFirst(uriString, WtcUri.URI_SCHEME_HTTP + "://", "");
                }
                uriString = WtcUri.URI_SCHEME_HTTPS + "://" + uriString;
            }
            else
            {
                if (uriString.startsWith(WtcUri.URI_SCHEME_HTTPS + "://"))
                {
                    uriString = WtcString.replaceFirst(uriString, WtcUri.URI_SCHEME_HTTPS + "://", "");
                }
                uriString = WtcUri.URI_SCHEME_HTTP + "://" + uriString;
            }
        }

        return uriString;
    }
}
