package com.twistpair.wave.thinclient;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Timer;
import java.util.TimerTask;
import java.util.Vector;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpErrorCodes;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.protocol.types.WtcpStringList;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcClientPhoneLineManager
{
    private static final String TAG                                 = WtcLog.TAG(WtcClientPhoneLineManager.class);

    public static final int     REACTIVATE_INTERVAL_SECONDS_DEFAULT = 30;

    private static class ReactivationTimerTask extends TimerTask
    {
        private static final String      TAG = WtcLog.TAG(ReactivationTimerTask.class);

        private final WtcClientPhoneLine mPhoneLine;

        public ReactivationTimerTask(WtcClientPhoneLine phoneLine)
        {
            mPhoneLine = phoneLine;
        }

        //@Override
        public void run()
        {
            WtcLog.info(TAG, "+run()");
            try
            {
                mPhoneLine.activate(true);
            }
            catch (Exception e)
            {
                WtcLog.error(TAG, "run: EXCEPTION", e);
            }
            WtcLog.info(TAG, "-run()");
        }
    }

    private final WtcClient         mClient;
    private final WtcClientListener mListener;

    private final Object            mSyncLock                               = new Object();

    private final Vector            mPhoneLineNumbers                       = new Vector();

    /**
     * Map&lt;String, WtcClientPhoneLine&gt;
     */
    private final Hashtable         mPhoneLines                             = new Hashtable();

    /**
     * Map&lt;Integer, String&gt
     */
    private final Hashtable         mLineActivationTransactionIdToPhoneLine = new Hashtable();

    /**
     * Map&lt;WtcClientPhoneLine, ReactivationTimerTask&gt;
     */
    private final Hashtable         mReactivationTimerTasks                 = new Hashtable();

    private Timer                   mReactivationTimer;

    private int                     mReactivateIntervalSeconds              = REACTIVATE_INTERVAL_SECONDS_DEFAULT;

    protected WtcClientPhoneLineManager(WtcClient client, WtcClientListener listener, //
                    WtcStack stack, //
                    WtcpStringList phoneLineNumbers)
    {
        if (client == null)
        {
            throw new IllegalArgumentException("client cannot be null");
        }

        if (listener == null)
        {
            throw new IllegalArgumentException("listener cannot be null");
        }

        if (stack == null)
        {
            throw new IllegalArgumentException("stack cannot be null");
        }

        if (phoneLineNumbers == null)
        {
            throw new IllegalArgumentException("phoneLineNumbers cannot be null");
        }

        mClient = client;
        mListener = listener;

        mReactivationTimer = new Timer();

        Enumeration enumerator = phoneLineNumbers.elements();
        String phoneLineNumber;
        WtcClientPhoneLine phoneLine;
        while (enumerator.hasMoreElements())
        {
            phoneLineNumber = (String) enumerator.nextElement();

            mPhoneLineNumbers.addElement(phoneLineNumber);

            phoneLine = new WtcClientPhoneLine(stack, this, phoneLineNumber);
            mPhoneLines.put(phoneLineNumber, phoneLine);
        }
    }

    protected void clear()
    {
        WtcLog.info(TAG, "+clear()");
        synchronized (mSyncLock)
        {
            if (mReactivationTimer != null)
            {
                mReactivationTimer.cancel();
                //mReactivationTimer.purge();
                mReactivationTimer = null;
            }

            mReactivationTimerTasks.clear();

            mPhoneLineNumbers.removeAllElements();

            /*
            WtcLog.info(TAG, "clear: BEGIN hanging up phone line call");
            Enumeration phoneLines = mPhoneLines.elements();
            WtcClientPhoneLine phoneLine;
            WtcClientPhoneCall phoneCall;
            while (phoneLines.hasMoreElements())
            {
                phoneLine = (WtcClientPhoneLine) phoneLines.nextElement();
                phoneCall = phoneLine.getPhoneCall();
                if (phoneCall != null)
                {
                    WtcLog.info(TAG, "clear: +phoneCall.hangup(); phoneCall=" + phoneCall);
                    phoneCall.hangup();
                    WtcLog.info(TAG, "clear: -phoneCall.hangup(); phoneCall=" + phoneCall);
                }
            }
            WtcLog.info(TAG, "clear: END hanging up phone line call");
            */
            mPhoneLines.clear();

            mLineActivationTransactionIdToPhoneLine.clear();
        }
        WtcLog.info(TAG, "-clear()");
    }

    /**
     * @return may be null
     */
    public WtcClientPhoneCallManager getPhoneCallManager()
    {
        return mClient.getPhoneCallManager();
    }

    public Enumeration getPhoneLineNumbers()
    {
        return mPhoneLineNumbers.elements();
    }

    public int getPhoneLineCount()
    {
        return mPhoneLineNumbers.size();
    }

    /**
     * @param index
     * @return null if no phone line exists at the given index
     */
    public WtcClientPhoneLine getPhoneLine(int index)
    {
        WtcClientPhoneLine phoneLine = null;
        synchronized (mSyncLock)
        {
            if (index >= 0 && index < mPhoneLineNumbers.size())
            {
                String phoneLineNumber = (String) mPhoneLineNumbers.elementAt(index);
                if (phoneLineNumber != null)
                {
                    phoneLine = getPhoneLine(phoneLineNumber);
                }
            }
        }
        return phoneLine;
    }

    /**
     * @param phoneLineNumber
     * @return null if no phone line exists with the given phone line number
     */
    public WtcClientPhoneLine getPhoneLine(String phoneLineNumber)
    {
        synchronized (mSyncLock)
        {
            return (WtcClientPhoneLine) mPhoneLines.get(phoneLineNumber);
        }
    }

    /**
     * @param intervalSeconds <= 0 to disable
     */
    public void setReactivateIntervalSeconds(int intervalSeconds)
    {
        synchronized (mSyncLock)
        {
            mReactivateIntervalSeconds = intervalSeconds;
        }
    }

    protected void reactivationTimerStart(WtcClientPhoneLine phoneLine)
    {
        WtcLog.info(TAG, "+reactivationTimerStart(phoneLine=" + phoneLine + ")");

        synchronized (mSyncLock)
        {
            if (mReactivateIntervalSeconds > 0)
            {
                reactivationTimerCancel(phoneLine);

                if (mReactivationTimer != null)
                {
                    ReactivationTimerTask task = new ReactivationTimerTask(phoneLine);
                    mReactivationTimerTasks.put(phoneLine, task);

                    int milliseconds = mReactivateIntervalSeconds * 1000;
                    WtcLog.warn(TAG, "reactivationTimerStart: +mReactivationTimer.schedule(task, " + milliseconds + ")");
                    mReactivationTimer.schedule(task, milliseconds);
                    WtcLog.warn(TAG, "reactivationTimerStart: -mReactivationTimer.schedule(task, " + milliseconds + ")");
                }
            }
        }

        WtcLog.info(TAG, "-reactivationTimerStart(phoneLine=" + phoneLine + ")");
    }

    protected void reactivationTimerCancel(WtcClientPhoneLine phoneLine)
    {
        WtcLog.info(TAG, "+reactivationTimerCancel(phoneLine=" + phoneLine + ")");

        synchronized (mSyncLock)
        {
            ReactivationTimerTask task = (ReactivationTimerTask) mReactivationTimerTasks.remove(phoneLine);
            if (task != null)
            {
                WtcLog.warn(TAG, "reactivationTimerCancel: +task.cancel()");
                task.cancel();
                WtcLog.warn(TAG, "reactivationTimerCancel: -task.cancel()");
            }
            else
            {
                WtcLog.warn(TAG, "mReactivationTimerTasks.remove(" + phoneLine + ") returned null!");
            }
        }

        WtcLog.info(TAG, "-reactivationTimerCancel(phoneLine=" + phoneLine + ")");
    }

    /**
     * NOTE: Currently only supports activating the <b>FIRST</b> phone line!
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(String[] phoneLines)
    {
        // TODO:(pv) Support multiple phone lines using logic similar to WtcClientChannelManager.doChannelOperation/onChannelOperation
        String phoneLineNumber = phoneLines[0];
        WtcClientPhoneLine phoneLine = getPhoneLine(phoneLineNumber);
        return (phoneLine == null) ? null : activate(phoneLine, true);
    }

    /**
     * NOTE: Currently only supports activating the <b>FIRST</b> phone line!
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcpStringList phoneLines)
    {
        // TODO:(pv) Support multiple phone lines using logic similar to WtcClientChannelManager.doChannelOperation/onChannelOperation
        String phoneLineNumber = (String) phoneLines.elementAt(0);
        WtcClientPhoneLine phoneLine = getPhoneLine(phoneLineNumber);
        return (phoneLine == null) ? null : phoneLine.activate(true);
    }

    /**
     * @param phoneLine
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcClientPhoneLine phoneLine, boolean on)
    {
        Integer transactionId = null;

        try
        {
            WtcLog.info(TAG, "+activate(phoneLine=" + phoneLine + ", on=" + on + ")");

            // TODO:(pv) Implement the ability to dynamically on/off any combination of our phone lines
            // using logic similar to WtcClientChannelManager.doChannelOperation/onChannelOperation

            phoneLine.mIsActivated = false;

            if (on)
            {
                String phoneLineNumber = phoneLine.getNumber();

                transactionId = phoneLine.getStack().sendPhoneLineSetActive(phoneLineNumber);
                if (transactionId != null)
                {
                    synchronized (mSyncLock)
                    {
                        mLineActivationTransactionIdToPhoneLine.put(transactionId, phoneLineNumber);
                    }
                }
            }
            else
            {
                throw new UnsupportedOperationException("phone line deactivation is currently not supported");
            }

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-activate(phoneLine=" + phoneLine + ", on=" + on + "); return " + transactionId);
        }
    }

    protected void onPhoneLinesSetActive(WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        String signature = "onPhoneLinesSetActive(..., errorCode=" + errorCode + ")";

        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            Integer transactionId = new Integer(controlHeader.transactionId);
            WtcClientPhoneLine phoneLine = null;
            synchronized (mSyncLock)
            {
                String phoneLineNumber = (String) mLineActivationTransactionIdToPhoneLine.remove(transactionId);
                phoneLine = getPhoneLine(phoneLineNumber);
            }

            boolean handled = false;
            if (phoneLine != null)
            {
                //
                // Always schedule a new reactivation in case we never get any onPhoneLineStatus
                //
                if (errorCode.isOK() || errorCode.getErrorCode() == WtcpErrorCodes.PhoneLineUnavailable)
                {
                    reactivationTimerStart(phoneLine);
                }

                if (mListener != null)
                {
                    handled = mListener.onPhoneLineActivating(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    phoneLine, errorCode);
                }
            }
            else
            {
                WtcLog.warn(TAG, "mLineActivationTransactionIdToPhoneLine.remove(" + transactionId + ") returned null!");

                // TODO:(pv) Support Unsolicited PhoneLineSetActive
            }

            if (!handled && mListener != null)
            {
                mListener.onPhoneLineSetActive(mClient, controlHeader.getOpType(), controlHeader.transactionId, //
                                errorCode);
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    protected void onPhoneLineStatus(WtcpControlHeader controlHeader, //
                    String phoneLineNumber, WtcpErrorCode errorCode)
    {
        String signature =
            "onPhoneLineStatus(..., phoneLineNumber=" + WtcString.quote(phoneLineNumber) + ", errorCode=" + errorCode + ")";

        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            WtcClientPhoneLine phoneLine = getPhoneLine(phoneLineNumber);

            boolean handled = false;
            if (phoneLine != null)
            {
                phoneLine.mIsActivated = isOK;

                if (isOK)
                {
                    reactivationTimerCancel(phoneLine);

                    if (mListener != null)
                    {
                        handled = mListener.onPhoneLineActivated(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneLine);
                    }
                }
                else
                {
                    if (errorCode.getErrorCode() == WtcpErrorCodes.PhoneLineUnavailable)
                    {
                        reactivationTimerStart(phoneLine);
                    }

                    if (mListener != null)
                    {
                        handled = mListener.onPhoneLineDeactivated(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneLine, errorCode);
                    }
                }
            }

            if (!handled && mListener != null)
            {
                mListener.onPhoneLineStatus(mClient, controlHeader.getOpType(), controlHeader.transactionId, //
                                phoneLineNumber, errorCode);
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }
}
