package com.twistpair.wave.thinclient;

import com.twistpair.wave.thinclient.WtcClientPhoneCall.CallTalkState;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpCallProgressState;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpCallType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallAnswer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallDtmf;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallHangup;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallOffer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallProgress;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcIntegerObjectMapPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcClientPhoneCallManager
{
    private static final String TAG = WtcLog.TAG(WtcClientPhoneCallManager.class);

    /**
     * Used during a Call Make Request to restore information that isn't available in the Response.<br>
     * This is used as a workaround for a limitation in the WTC Protocol.
     */
    protected static class CallMakeWrapper
    {
        public final WtcClientPhoneLine mPhoneLine;
        public final byte               mCallType;
        public final String             mRemoteName;

        public CallMakeWrapper(WtcClientPhoneLine phoneLine, byte callType, String remoteName)
        {
            mPhoneLine = phoneLine;
            mCallType = callType;
            mRemoteName = remoteName;
        }
    }

    /**
     * Used during a Call DTMF Request to restore information that isn't available in the Response<br>
     * This is used as a workaround for a limitation in the WTC Protocol.
     */
    protected static class CallDtmfWrapper
    {
        public final WtcClientPhoneCall mPhoneCall;
        public final String             mDigits;

        public CallDtmfWrapper(WtcClientPhoneCall phoneCall, String digits)
        {
            mPhoneCall = phoneCall;
            mDigits = digits;
        }
    }

    private final WtcClientListener           mListener;
    private final WtcClient                   mClient;

    private final Object                      mSyncLock                               = new Object();

    /**
     * Map&lt;Integer, {@link WtcClientPhoneCall}&gt;
     */
    private final WtcIntegerObjectMapPlatform mPhoneCalls                             = new WtcIntegerObjectMapPlatform();

    /**
     * Map&lt;Integer, {@link CallMakeWrapper}&gt;
     */
    private final WtcIntegerObjectMapPlatform mCallMakeTransactionIdToCallMakeWrapper = new WtcIntegerObjectMapPlatform();

    /**
     * Map&lt;Integer, {@link CallDtmfWrapper}&gt;
     */
    private final WtcIntegerObjectMapPlatform mCallDtmfTransactionIdToCallDtmfWrapper = new WtcIntegerObjectMapPlatform();

    /**
     * We currently only support one phone call being active/connected at a time.
     */
    private WtcClientPhoneCall                mPhoneCallConnected;

    protected WtcClientPhoneCallManager(WtcClient client, WtcClientListener listener)
    {
        if (client == null)
        {
            throw new IllegalArgumentException("client cannot be null");
        }

        if (listener == null)
        {
            throw new IllegalArgumentException("listener cannot be null");
        }

        mClient = client;
        mListener = listener;
    }

    protected void clear()
    {
        WtcLog.info(TAG, "+clear()");
        synchronized (mSyncLock)
        {
            /*
            WtcLog.info(TAG, "clear: BEGIN hanging up phone calls");
            Enumeration phoneCalls = mPhoneCalls.elements();
            WtcClientPhoneCall phoneCall;
            while (phoneCalls.hasMoreElements())
            {
                phoneCall = (WtcClientPhoneCall) phoneCalls.nextElement();
                WtcLog.info(TAG, "clear: +phoneCall.hangup(); phoneCall=" + phoneCall);
                phoneCall.hangup();
                WtcLog.info(TAG, "clear: -phoneCall.hangup(); phoneCall=" + phoneCall);
            }
            WtcLog.info(TAG, "clear: END hanging up phone calls");
            */
            mPhoneCalls.clear();
            mPhoneCallConnected = null;

            mCallMakeTransactionIdToCallMakeWrapper.clear();
            mCallDtmfTransactionIdToCallDtmfWrapper.clear();
        }
        WtcLog.info(TAG, "-clear()");
    }

    /**
     * @return true if there are any phone calls active, being made, or being received, otherwise false
     */
    public boolean getIsCallInProgressOrConnected()
    {
        synchronized (mSyncLock)
        {
            return mPhoneCallConnected != null || mPhoneCalls.size() > 0 || mCallMakeTransactionIdToCallMakeWrapper.size() > 0;
        }
    }

    /**
     * @return null if there is no connected call
     */
    public WtcClientPhoneCall getConnectedPhoneCall()
    {
        synchronized (mSyncLock)
        {
            return mPhoneCallConnected;
        }
    }

    /**
     * @param callId
     * @return null if the specified callId does not exist 
     */
    private WtcClientPhoneCall getPhoneCall(int callId)
    {
        synchronized (mSyncLock)
        {
            return (WtcClientPhoneCall) mPhoneCalls.get(callId);
        }
    }

    //
    // Call Make
    //

    /**
     * @param callType One of WtcpCallType.*
     * @param phoneLineNumber
     * @param remoteNumber
     * @param remoteName
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer call(byte callType, String phoneLineNumber, String remoteNumber, String remoteName)
    {
        Integer transactionId = null;

        WtcClientPhoneLineManager lineManager = mClient.getPhoneLineManager();
        if (lineManager != null)
        {
            WtcClientPhoneLine phoneLine = lineManager.getPhoneLine(phoneLineNumber);
            if (phoneLine != null)
            {
                transactionId = call(phoneLine, callType, remoteNumber, remoteName);
            }
        }

        return transactionId;
    }

    /**
     * @param phoneLine
     * @param callType One of WtcpCallType.*
     * @param remoteNumber Remote phone number or endpoint id (depending on callType)
     * @param remoteName
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer call(WtcClientPhoneLine phoneLine, byte callType, String remoteNumber, String remoteName)
    {
        Integer transactionId = null;

        String signature =
            "call(phoneLine=" + phoneLine + ", callType=" + WtcpCallType.toString(callType) + ", remoteNumber="
                            + WtcString.quote(remoteNumber) + ", remoteName=" + WtcString.quote(remoteName) + ")";

        try
        {
            WtcLog.info(TAG, "+" + signature);

            transactionId = phoneLine.getStack().sendCallMake(callType, phoneLine.getNumber(), remoteNumber);
            if (transactionId != null)
            {
                CallMakeWrapper callMakeWrapper = new CallMakeWrapper(phoneLine, callType, remoteName);
                synchronized (mSyncLock)
                {
                    mCallMakeTransactionIdToCallMakeWrapper.put(transactionId.intValue(), callMakeWrapper);
                }
            }

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-" + signature + "; return " + transactionId);
        }
    }

    /**
     * NOTE: Neat trick that this can be Unsolicited! 
     * @param controlHeader
     * @param callInfo
     */
    public void onCallMake(WtcpControlHeader controlHeader, //
                    WtcpCallInfo callInfo)
    {
        String signature = "onCallMake(..., callInfo=" + callInfo + ")";

        boolean isOK = callInfo.errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            int transactionId = controlHeader.transactionId;
            CallMakeWrapper callMakeWrapper;
            synchronized (mSyncLock)
            {
                callMakeWrapper = (CallMakeWrapper) mCallMakeTransactionIdToCallMakeWrapper.remove(transactionId);
            }

            boolean handled = false;
            if (callMakeWrapper != null)
            {
                if (isOK)
                {
                    //
                    // Create new WtcClientPhoneCall and add to mPhoneCalls map 
                    //

                    WtcClientPhoneCall phoneCall = null;
                    int callId = callInfo.callId;
                    phoneCall = new WtcClientPhoneCall(this, callMakeWrapper, callInfo);
                    synchronized (mSyncLock)
                    {
                        mPhoneCalls.put(callId, phoneCall);
                    }

                    if (mListener != null)
                    {
                        handled = mListener.onCallMake(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneCall);
                    }
                }
                else
                {
                    if (mListener != null)
                    {
                        handled = mListener.onCallMakeError(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        callMakeWrapper.mCallType, callMakeWrapper.mPhoneLine, //
                                        callInfo.remoteNumber, callMakeWrapper.mRemoteName, //
                                        callInfo.errorCode);
                    }
                }
            }
            else
            {
                WtcLog.error(TAG, "mCallMakeTransactionIdToCallMakeInfo.remove(" + transactionId + ") returned null!");

                // TODO:(pv) Support Unsolicited CallMake
            }

            if (!handled && mListener != null)
            {
                mListener.onCallMake(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                callInfo);
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    //
    // Call Progress
    //

    /**
     * NOTE: Always Unsolicited! 
     * @param controlHeader
     * @param callProgress
     */
    public void onCallProgress(WtcpControlHeader controlHeader, //
                    WtcpCallProgress callProgress)
    {

        try
        {
            WtcLog.info(TAG, "+onCallProgress(..., callProgress=" + callProgress + ")");

            int callId = callProgress.callId;

            boolean handled = false;
            WtcClientPhoneCall phoneCall = getPhoneCall(callId);
            if (phoneCall != null)
            {
                phoneCall.setCallProgressState(callProgress.progress);

                if (mListener != null)
                {
                    switch (callProgress.progress)
                    {
                        case WtcpCallProgressState.Proceeding:
                            handled = mListener.onCallProceeding(mClient, //
                                            controlHeader.getOpType(), controlHeader.transactionId, //
                                            phoneCall);
                            break;
                        case WtcpCallProgressState.Ringing:
                            handled = mListener.onCallRinging(mClient, //
                                            controlHeader.getOpType(), controlHeader.transactionId, //
                                            phoneCall);
                            break;
                        case WtcpCallProgressState.Connected:
                            synchronized (mSyncLock)
                            {
                                mPhoneCallConnected = phoneCall;
                            }
                            handled = mListener.onCallConnected(mClient, //
                                            controlHeader.getOpType(), controlHeader.transactionId, //
                                            phoneCall);
                            break;
                        default:
                            WtcLog.error(TAG, "onCallProgress: UNKNOWN progress=" + callProgress.progress);
                            break;
                    }
                }
            }

            if (!handled && mListener != null)
            {
                mListener.onCallProgress(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                callProgress);
            }
        }
        finally
        {
            WtcLog.info(TAG, "-onCallProgress(..., callProgress=" + callProgress + ")");
        }
    }

    //
    // Call Offer
    //

    /**
     * NOTE: Always Unsolicited! 
     * @param controlHeader
     * @param callOffer
     */
    protected void onCallOffer(WtcpControlHeader controlHeader, //
                    WtcpCallOffer callOffer)
    {
        try
        {
            WtcLog.info(TAG, "+onCallOffer(..., callOffer=" + callOffer + ")");

            int callId = callOffer.callId;

            //
            // Create new WtcClientPhoneCall and add to mPhoneCalls
            //

            WtcClientPhoneCall phoneCall = null;

            WtcClientPhoneLineManager lineManager = mClient.getPhoneLineManager();
            if (lineManager != null)
            {
                String phoneLineNumber = callOffer.toNumber;
                WtcClientPhoneLine phoneLine = lineManager.getPhoneLine(phoneLineNumber);
                if (phoneLine != null)
                {
                    phoneCall = new WtcClientPhoneCall(this, phoneLine, callOffer);
                    synchronized (mSyncLock)
                    {
                        mPhoneCalls.put(callId, phoneCall);
                    }
                }
            }

            boolean handled = false;
            if (phoneCall != null)
            {
                // Nothing to do?

                if (mListener != null)
                {
                    handled = mListener.onCallOffer(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    phoneCall);
                }
            }

            if (!handled && mListener != null)
            {
                mListener.onCallOffer(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                callOffer);
            }
        }
        finally
        {
            WtcLog.info(TAG, "-onCallOffer(..., callOffer=" + callOffer + ")");
        }
    }

    //
    // Call Answer
    //

    protected Integer answer(int callId)
    {
        WtcClientPhoneCall phoneCall = getPhoneCall(callId);
        return (phoneCall == null) ? null : answer(phoneCall);
    }

    public Integer answer(WtcClientPhoneCall phoneCall)
    {
        Integer transactionId = null;

        try
        {
            WtcLog.info(TAG, "+answer(phoneCall=" + phoneCall + ")");

            transactionId = phoneCall.getStack().sendCallAnswer(phoneCall.getCallId());

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-answer(phoneCall=" + phoneCall + "); return " + transactionId);
        }
    }

    /**
     * NOTE: Neat trick that this can be Unsolicited! 
     * @param controlHeader
     * @param callAnswer
     */
    protected void onCallAnswer(WtcpControlHeader controlHeader, //
                    WtcpCallAnswer callAnswer)
    {
        String signature = "onCallAnswer(..., callAnswer=" + callAnswer + ")";

        WtcpErrorCode errorCode = callAnswer.errorCode;
        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            int callId = callAnswer.callId;

            boolean handled = false;
            WtcClientPhoneCall phoneCall = getPhoneCall(callId);
            if (phoneCall != null)
            {
                if (isOK)
                {
                    // Nothing to do here; CallAnswer success will result in onCallProgress(Connected)

                    if (mListener != null)
                    {
                        handled = mListener.onCallAnswered(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneCall);
                    }
                }
                else
                {
                    if (mListener != null)
                    {
                        handled = mListener.onCallAnswerError(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneCall, errorCode);
                    }
                }
            }

            if (!handled && mListener != null)
            {
                mListener.onCallAnswer(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                callAnswer);
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    //
    // Call Hang Up
    //

    public Integer hangup(int callId)
    {
        WtcClientPhoneCall phoneCall = getPhoneCall(callId);
        return (phoneCall == null) ? null : hangup(phoneCall);
    }

    public Integer hangup(WtcClientPhoneCall phoneCall)
    {
        Integer transactionId = null;

        try
        {
            WtcLog.info(TAG, "+hangup(phoneCall=" + phoneCall + ")");

            int callId = phoneCall.getCallId();

            phoneCall.onCallHangup();

            transactionId = phoneCall.getStack().sendCallHangup(callId);
            if (transactionId == null)
            {
                // TODO:(pv) I am seeing a "NullPointerException" thrown when unboxing a null transactionId in onCallHangup
                transactionId = new Integer(-1);
            }

            //
            // Always handle the case where the client could be already disconnected, or may disconnect before RXing a Response/Error.
            // Assume success and always directly fire event of solicited hangup here.
            // Ignore Response/Error in onCallHangup to avoid double firing the event.
            // Only fire the event in onCallHangup if the RXed message is Unsolicited.
            //

            onCallHangup(callId);

            if (mListener != null)
            {
                boolean handled = mListener.onCallHangup(mClient, //
                                WtcpOpType.Request, transactionId.intValue(), //
                                phoneCall);
                if (!handled)
                {
                    //
                    // Fake a success; the only error case is that callId doesn't exist.
                    // If that is the case then the call is already hung up.
                    // ie: This will always effectively succeed!
                    //
                    WtcpCallHangup callHangup = new WtcpCallHangup(callId, WtcpErrorCode.OK);
                    mListener.onCallHangup(mClient, //
                                    WtcpOpType.Request, transactionId.intValue(), //
                                    callHangup);
                }
            }

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-hangup(phoneCall=" + phoneCall + "); return " + transactionId);
        }
    }

    /**
     * Always removes any hung up call from mPhoneCalls.<br>
     * Sets mPhoneCallConnected to null if this was our connected call.<br>
     * ie: Doesn't set mPhoneCallConnected to null if we hung up an offered call.
     * @param callId
     * @return null if a call with the specified callId does not exist
     */
    private WtcClientPhoneCall onCallHangup(int callId)
    {
        WtcClientPhoneCall phoneCall;

        synchronized (mSyncLock)
        {
            phoneCall = (WtcClientPhoneCall) mPhoneCalls.remove(callId);

            if (mPhoneCallConnected != null)
            {
                if (mPhoneCallConnected.getCallId() == callId)
                {
                    mPhoneCallConnected = null;
                }
            }
        }

        return phoneCall;
    }

    /**
     * NOTE: Most definitely can be Unsolicited! 
     * @param controlHeader
     * @param callHangup
     */
    protected void onCallHangup(WtcpControlHeader controlHeader, //
                    WtcpCallHangup callHangup)
    {
        String signature = "onCallHangup(..., callHangup=" + callHangup + ")";

        WtcpErrorCode errorCode = callHangup.errorCode;
        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            int callId = callHangup.callId;

            //
            // Solicited hangups directly fire onCallHangup to make sure the event fires even if
            // the client is already disconnected, or disconnects before RXing a Response/Error.
            // Ignore onCallHangup Response/Error here to avoid double firing the event.
            // Only fire the event here if the RXed message is Unsolicited.
            //

            WtcClientPhoneCall phoneCall = onCallHangup(callId);

            if (controlHeader.isUnsolicited())
            {
                boolean handled = false;
                if (phoneCall != null)
                {
                    //
                    // We found the phoneCall in mPhoneCalls.
                    // That means we didn't locally call hangup().
                    // Thus, this must be an Unsolicited hangup().
                    // Either the other Client or the Server hung up on us.
                    //

                    phoneCall.onCallHangup();

                    if (mListener != null)
                    {
                        handled = mListener.onCallHangup(mClient, //
                                        controlHeader.getOpType(), controlHeader.transactionId, //
                                        phoneCall);
                    }
                }

                if (!handled && mListener != null)
                {
                    mListener.onCallHangup(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    callHangup);
                }
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    //
    // Call DTMF
    //

    public Integer dtmf(int callId, String digits)
    {
        WtcClientPhoneCall phoneCall = getPhoneCall(callId);
        return (phoneCall == null) ? null : dtmf(phoneCall, digits);
    }

    public Integer dtmf(WtcClientPhoneCall phoneCall, String digits)
    {
        Integer transactionId = null;

        try
        {
            WtcLog.info(TAG, "+dtmf(phoneCall=" + phoneCall + ", digits=" + WtcString.quote(digits) + ")");

            transactionId = phoneCall.getStack().sendCallDtmf(phoneCall.getCallId(), digits);
            if (transactionId != null)
            {
                CallDtmfWrapper callDtmfWrapper = new CallDtmfWrapper(phoneCall, digits);
                synchronized (mSyncLock)
                {
                    mCallDtmfTransactionIdToCallDtmfWrapper.put(transactionId.intValue(), callDtmfWrapper);
                }
            }

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-dtmf(phoneCall=" + phoneCall + ", digits=" + WtcString.quote(digits) + "); return "
                            + transactionId);
        }
    }

    /**
     * NOTE: Neat trick that this can be Unsolicited! 
     * @param controlHeader
     * @param errorCode null for success
     */
    protected void onCallDtmf(WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        String signature = "onCallDtmf(..., errorCode=" + errorCode + ")";

        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            int transactionId = controlHeader.transactionId;
            CallDtmfWrapper callDtmfWrapper;
            synchronized (mSyncLock)
            {
                callDtmfWrapper = (CallDtmfWrapper) mCallDtmfTransactionIdToCallDtmfWrapper.remove(transactionId);
            }

            boolean handled = false;
            if (callDtmfWrapper != null)
            {
                // Nothing to do?

                if (mListener != null)
                {
                    handled = mListener.onCallDtmf(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    callDtmfWrapper.mPhoneCall, callDtmfWrapper.mDigits, errorCode);
                }
            }
            else
            {
                WtcLog.error(TAG, "mCallDtmfTransactionIdToCallDtmfInfo.remove(" + transactionId + ") returned null!");
            }

            if (!handled && mListener != null)
            {
                mListener.onCallDtmf(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                errorCode);
            }
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    /**
     * NOTE: Most definitely can be Unsolicited!<br>
     * Always a Success condition, Never an Error condition. 
     * @param controlHeader
     * @param callDtmf
     */
    protected void onCallDtmf(WtcpControlHeader controlHeader, //
                    WtcpCallDtmf callDtmf)
    {
        try
        {
            WtcLog.info(TAG, "+onCallDtmf(..., callDtmf=" + callDtmf + ")");

            int callId = callDtmf.callId;

            boolean handled = false;
            WtcClientPhoneCall phoneCall = getPhoneCall(callId);
            if (phoneCall != null)
            {
                phoneCall.onCallDtmf(controlHeader, callDtmf);

                if (mListener != null)
                {
                    handled = mListener.onCallDtmf(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    phoneCall, callDtmf.digits);
                }
            }

            if (!handled && mListener != null)
            {
                mListener.onCallDtmf(mClient, //
                                controlHeader.getOpType(), controlHeader.transactionId, //
                                callDtmf);
            }
        }
        finally
        {
            WtcLog.info(TAG, "-onCallDtmf(..., callDtmf=" + callDtmf + ")");
        }
    }

    //
    // Call Push To Talk
    //

    public Integer pushToTalk(int callId, boolean on)
    {
        WtcClientPhoneCall phoneCall = getPhoneCall(callId);
        return (phoneCall == null) ? null : pushToTalk(phoneCall, on);
    }

    public Integer pushToTalk(WtcClientPhoneCall phoneCall, boolean on)
    {
        Integer transactionId = null;

        try
        {
            WtcLog.info(TAG, "+pushToTalk(phoneCall=" + phoneCall + ", on=" + on + ")");

            transactionId = phoneCall.getStack().sendCallPushToTalk(phoneCall.getCallId(), on);

            if (transactionId != null && on)
            {
                phoneCall.setCallTalkState(CallTalkState.ONING);
            }
            else
            {
                //
                // Jump straight to the "OFF" state.
                // The only error case is that callId doesn't exist.
                // If that is the case then Call PTT is already OFF.
                // ie: Jump straight to the "OFF" state.
                //
                phoneCall.setCallTalkState(CallTalkState.OFF);
            }

            return transactionId;
        }
        finally
        {
            WtcLog.info(TAG, "-pushToTalk(phoneCall=" + phoneCall + ", on=" + on + "); return " + transactionId);
        }
    }

    /**
     * NOTE: Neat trick that this can be Unsolicited! 
     * @param controlHeader
     * @param callId
     * @param errorCode
     * @return true if handled
     */
    protected boolean onCallPushToTalkOn(WtcpControlHeader controlHeader, //
                    WtcInt32 callId, WtcpErrorCode errorCode)
    {
        boolean handled = false;

        String signature = "onCallPushToTalkOn(..., callId=" + callId + ", errorCode=" + errorCode + ")";

        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            WtcClientPhoneCall phoneCall = getPhoneCall(callId.value);
            if (phoneCall != null)
            {
                phoneCall.setCallTalkState(errorCode.isOK() ? CallTalkState.ON : CallTalkState.OFF);

                if (mListener != null)
                {
                    handled = mListener.onCallPushToTalkOn(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    phoneCall, errorCode);
                }
            }
            else
            {
                WtcLog.warn(TAG, "onCallPushToTalkOn: getPhoneCall(" + callId.value + ") returned null!");

                if (mListener != null)
                {
                    handled = mListener.onCallPushToTalkOn(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    callId, errorCode);
                }
            }

            return handled;
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }

    /**
     * NOTE: Most definitely can be Unsolicited! 
     * @param controlHeader
     * @param callId
     * @param errorCode
     * @return true if handled
     */
    protected boolean onCallPushToTalkOff(WtcpControlHeader controlHeader, //
                    WtcInt32 callId, WtcpErrorCode errorCode)
    {
        boolean handled = false;

        String signature = "onCallPushToTalkOff(..., callId=" + callId + ", errorCode=" + errorCode + ")";

        boolean isOK = errorCode.isOK();

        try
        {
            if (isOK)
            {
                WtcLog.info(TAG, "+" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "+" + signature);
            }

            WtcClientPhoneCall phoneCall = getPhoneCall(callId.value);
            if (phoneCall != null)
            {
                phoneCall.setCallTalkState(CallTalkState.OFF);

                if (mListener != null)
                {
                    handled = mListener.onCallPushToTalkOff(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    phoneCall, errorCode);
                }
            }
            else
            {
                WtcLog.warn(TAG, "onCallPushToTalkOff: getPhoneCall(" + callId.value + ") returned null!");

                if (mListener != null)
                {
                    handled = mListener.onCallPushToTalkOff(mClient, //
                                    controlHeader.getOpType(), controlHeader.transactionId, //
                                    callId, errorCode);
                }
            }

            return handled;
        }
        finally
        {
            if (isOK)
            {
                WtcLog.info(TAG, "-" + signature);
            }
            else
            {
                WtcLog.warn(TAG, "-" + signature);
            }
        }
    }
}
