package com.twistpair.wave.thinclient;

import com.twistpair.wave.thinclient.WtcClientPhoneCallManager.CallMakeWrapper;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpCallProgressState;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpCallType;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallDtmf;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallOffer;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcClientPhoneCall
{
    private static final String TAG = WtcLog.TAG(WtcClientPhoneCall.class);

    public static abstract class CallTalkState
    {
        public static final int OFF    = 0;
        public static final int ONING  = 1;
        public static final int ON     = 2;
        public static final int OFFING = 3;

        public static String getName(int talkState)
        {
            switch (talkState)
            {
                case OFF: // 0
                    return "OFF";
                case ONING: // 1
                    return "ONING";
                case ON: // 2
                    return "ON";
                case OFFING: // 3
                    return "OFFING";
            }
            return "?";
        }

        public static String toString(int talkState)
        {
            return toString(talkState, 'd');
        }

        public static String toString(int talkState, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(talkState) + "\"(" + WtcString.formatNumber(talkState, 1) + ')';
                default:
                    return '\"' + getName(talkState) + "\"(0x" + WtcString.toHexString(talkState, 2) + ')';
            }
        }
    }

    public static final String              CALL_TYPE_ENDPOINT_PREFIX = "EP#";

    private final WtcClientPhoneCallManager mCallManager;

    private final byte                      mCallType;
    private final WtcClientPhoneLine        mPhoneLine;
    private final String                    mRemoteName;

    private final int                       mCallId;
    private final String                    mRemoteNumber;

    private final boolean                   mIsOutgoingCall;

    private final Object                    mSyncState                = new Object();
    private int                             mCallProgressState        = WtcpCallProgressState.Disconnected;
    private int                             mCallTalkState            = CallTalkState.OFF;

    /**
     * Instance created by a Call Make
     * @param callManager
     * @param callMakeInfo
     * @param callInfo
     */
    protected WtcClientPhoneCall(WtcClientPhoneCallManager callManager, //
                    CallMakeWrapper callMakeInfo, WtcpCallInfo callInfo)
    {
        if (callManager == null)
        {
            throw new IllegalArgumentException("callManager cannot be null");
        }

        if (callMakeInfo == null)
        {
            throw new IllegalArgumentException("callMakeInfo cannot be null");
        }

        if (callMakeInfo.mPhoneLine == null)
        {
            throw new IllegalArgumentException("callMakeInfo.mPhoneLine cannot be null");
        }

        if (callInfo == null)
        {
            throw new IllegalArgumentException("callInfo cannot be null");
        }

        mCallManager = callManager;

        mPhoneLine = callMakeInfo.mPhoneLine;
        mPhoneLine.setPhoneCall(this);

        mCallType = callMakeInfo.mCallType;
        mRemoteName = callMakeInfo.mRemoteName;

        mCallId = callInfo.callId;
        mRemoteNumber = callInfo.remoteNumber;
        mIsOutgoingCall = true;
    }

    /**
     * Instance created by a Call Offer
     * @param callManager
     * @param phoneLine
     * @param callOffer
     */
    protected WtcClientPhoneCall(WtcClientPhoneCallManager callManager, //
                    WtcClientPhoneLine phoneLine, WtcpCallOffer callOffer)
    {
        if (callManager == null)
        {
            throw new IllegalArgumentException("callManager cannot be null");
        }

        if (phoneLine == null)
        {
            throw new IllegalArgumentException("phoneLine cannot be null");
        }

        if (callOffer == null)
        {
            throw new IllegalArgumentException("callOffer cannot be null");
        }

        mCallManager = callManager;

        mPhoneLine = phoneLine;
        mPhoneLine.setPhoneCall(this);

        String remoteName = callOffer.fromName;
        if (remoteName.startsWith(CALL_TYPE_ENDPOINT_PREFIX))
        {
            mCallType = WtcpCallType.Endpoint;
            mRemoteName = remoteName.substring(CALL_TYPE_ENDPOINT_PREFIX.length());
        }
        else
        {
            mCallType = WtcpCallType.E164;
            mRemoteName = remoteName;
        }

        mCallId = callOffer.callId;
        mRemoteNumber = callOffer.fromNumber;
        mIsOutgoingCall = false;
    }

    //@Override
    public String toString()
    {
        return new StringBuffer() //
        .append('{') //
        .append("mPhoneLine=").append(mPhoneLine) //
        .append(", mCallId=").append(mCallId) //
        .append(", mCallType=").append(WtcpCallType.toString(mCallType)) //
        .append(", mRemoteName=").append(WtcString.quote(mRemoteName)) //
        .append(", mRemoteNumber=").append(WtcString.quote(mRemoteNumber)) //
        .append(", mIsOutgoingCall=").append(mIsOutgoingCall) //
        .append(", mCallProgressState=").append(WtcpCallProgressState.toString(mCallProgressState)) //
        .append(", mCallTalkState=").append(CallTalkState.toString(mCallTalkState)) //
        .append('}') //
        .toString();
    }

    //@Override
    public boolean equals(Object o)
    {
        if (o != null && o instanceof WtcClientPhoneCall)
        {
            return getCallId() == ((WtcClientPhoneCall) o).getCallId();
        }
        return false;
    }

    /**
     * @return never null
     */
    protected WtcStack getStack()
    {
        return mPhoneLine.getStack();
    }

    public int getCallId()
    {
        return mCallId;
    }

    /**
     * @return One of WtcpCallType.* 
     */
    public byte getCallType()
    {
        return mCallType;
    }

    public WtcClientPhoneLine getPhoneLine()
    {
        return mPhoneLine;
    }

    public String getRemoteName()
    {
        return mRemoteName;
    }

    public String getRemoteNumber()
    {
        return mRemoteNumber;
    }

    public boolean getIsOutgoingCall()
    {
        return mIsOutgoingCall;
    }

    public boolean getIsCallConnected()
    {
        synchronized (mSyncState)
        {
            return mCallProgressState == WtcpCallProgressState.Connected;
        }
    }

    /**
     * @return One of WtcpCallProgressState.*
     */
    public int getCallProgressState()
    {
        synchronized (mSyncState)
        {
            return mCallProgressState;
        }
    }

    /**
     * @param progressState One of WtcpCallProgressState.*
     */
    protected void setCallProgressState(int progressState)
    {
        synchronized (mSyncState)
        {
            WtcLog.info(TAG, "setCallProgressState(" + WtcpCallProgressState.toString(progressState) + ")");
            mCallProgressState = progressState;
        }
    }

    /**
     * @return One of CallTalkState.*
     */
    public int getCallTalkState()
    {
        synchronized (mSyncState)
        {
            return mCallTalkState;
        }
    }

    /**
     * Called by onCallPushToTalkOn(...), onCallPushToTalkOff(...), and WtcClientPhoneCallManager.pushToTalk(...)
     * @param talkState One of CallTalkState.*
     */
    protected void setCallTalkState(int talkState)
    {
        synchronized (mSyncState)
        {
            WtcLog.info(TAG, "setCallTalkState(" + CallTalkState.toString(talkState) + ")");
            mCallTalkState = talkState;
        }
    }

    public Integer answer()
    {
        return mCallManager.answer(this);
    }

    public Integer hangup()
    {
        return mCallManager.hangup(this);
    }

    /**
     * Sets call progress to disconnected state, stops talking on a call, and sets this call's phoneline's call to null.<br> 
     * Called by <b>solicited</b> WtcClientPhoneCallManager.hangup(...) Request<br>
     * and <b>Unsolicited</b> WtcClientPhoneCallManager.onCallHangup(...)
     */
    protected void onCallHangup()
    {
        WtcLog.info(TAG, "+onCallHangup()");

        synchronized (mSyncState)
        {
            setCallProgressState(WtcpCallProgressState.Disconnected);

            /*
            // TODO:(pv) I expect the server to do this for us automatically...
            //      Do we need to do this to heal/reset any local client state?
            if (mCallTalkState != CallTalkState.OFF)
            {
                pushToTalk(false);
            }
            */

            mPhoneLine.setPhoneCall(null);
        }

        WtcLog.info(TAG, "-onCallHangup()");
    }

    public Integer dtmf(String digits)
    {
        return mCallManager.dtmf(this, digits);
    }

    /**
     * NOTE: Most definitely can be Unsolicited!
     * @param controlHeader
     * @param callDtmf
     */
    protected void onCallDtmf(WtcpControlHeader controlHeader, //
                    WtcpCallDtmf callDtmf)
    {
        WtcLog.info(TAG, "+onCallDtmf(..., callDtmf=" + callDtmf + ")");

        // TODO:(pv) Store callDtmf.digits in to a queue that can be enumerated

        WtcLog.info(TAG, "-onCallDtmf(..., callDtmf=" + callDtmf + ")");
    }

    public Integer pushToTalk(boolean on)
    {
        return mCallManager.pushToTalk(this, on);
    }
}
