package com.twistpair.wave.thinclient;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.NoSuchElementException;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpChannelChange;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpChannelFlags;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpCode;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelActivity;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdErrorDictionary;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdList;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.protocol.types.WtcpKeyValueList;
import com.twistpair.wave.thinclient.protocol.types.WtcpStringList;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcIntegerObjectMapPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcClientChannelManager
{
    private static final String  TAG                = WtcLog.TAG(WtcClientChannelManager.class);

    /**
     * EXPERIMENTAL way to mute the channels before/while they are being activated
     */
    public boolean               AUTO_MUTE_CHANNELS = false;

    private static final boolean VERBOSE_LOG        = false;

    public static class ChannelOperationType
    {
        public static final int OFF      = 0;
        public static final int ON       = 1;
        public static final int ABSOLUTE = 2;

        public static String toString(int opType)
        {
            switch (opType)
            {
                case OFF:
                    return "OFF";
                case ON:
                    return "ON";
                case ABSOLUTE:
                    return "ABSOLUTE";
                default:
                    return "{UNKNOWN}";
            }
        }
    }

    /**
     * Hashtable&lt;WtcInt32, WtcClientChannel&gt;
     */
    private final Hashtable         mChannelsAll          = new Hashtable();

    /**
     * A special collection that can be used by a multi-channel app to decide if the mic should be closed or not.<br>
     * ie: If no channels are being talked on then app should close the mic.
     */
    private final WtcpChannelIdList mChannelsPttOningOrOn = new WtcpChannelIdList();

    private final WtcClient         mClient;

    private boolean                 mReceivedChannels;

    public WtcClientChannelManager(WtcClient client)
    {
        if (client == null)
        {
            throw new IllegalArgumentException("client cannot be null");
        }

        mClient = client;
    }

    protected void clear()
    {
        synchronized (mChannelsAll)
        {
            mChannelsAll.clear();
        }
    }

    public boolean hasReceivedChannels()
    {
        return mReceivedChannels;
    }

    /**
     * Should only be called by WtcClient.onSetCredentialsResponse(...)
     */
    protected void setChannels(WtcpChannelInfoList channels)
    {
        WtcLog.info(TAG, "setChannels(...): Setting channels " + channels);

        synchronized (mChannelsAll)
        {
            mReceivedChannels = true;

            mChannelsAll.clear();

            Enumeration channelInfos = channels.elements();
            while (channelInfos.hasMoreElements())
            {
                WtcpChannelInfo channelInfo = (WtcpChannelInfo) channelInfos.nextElement();

                // TODO:(grava) Channel active/mute/pushed state should persist across sessions (Request from client or Unsolicited)

                WtcClientChannel channel = new WtcClientChannel(this, channelInfo);
                mChannelsAll.put(WtcInt32.valueOf(channelInfo.id, true), channel);
            }
        }
    }

    /**
     * Should only be called by unsolicited WtcClient.onChannelList(...)
     */
    /*
    protected void addChannels(WtcpChannelInfoList channels)
    {
        synchronized (channelsAll)
        {
            // TODO:(pv) Implement this after grava implements Unsolicited Channel Updates
        }
    }
    */

    /**
     * @return a read-only collection of WtcpChannelInfo
     */
    public Enumeration getChannels()
    {
        synchronized (mChannelsAll)
        {
            return mChannelsAll.elements();
        }
    }

    // TODO:(pv) Change this to "getTalkingCount()" logic
    public boolean isTalkingOnAnyChannel()
    {
        // intentionally synchronized on channelsAll and updated in doChannelOperationRequest
        synchronized (mChannelsAll)
        {
            return mChannelsPttOningOrOn.size() > 0;
        }
    }

    public WtcpChannelIdList getChannelIds()
    {
        synchronized (mChannelsAll)
        {
            WtcpChannelIdList channelIds = new WtcpChannelIdList();
            Enumeration channels = mChannelsAll.elements();
            while (channels.hasMoreElements())
            {
                channelIds.addElement(((WtcClientChannel) channels.nextElement()).getIdSession());
            }
            return channelIds;
        }
    }

    /**
     * @param id id of a channel
     * @return the WtcClientChannel with the given id, or null if the specified id does not exist.
     */
    public WtcClientChannel getChannel(int id)
    {
        return getChannel(WtcInt32.valueOf(id, true));
    }

    /**
     * @param id id of a channel
     * @return the WtcClientChannel with the given id, or null if the specified id does not exist.
     */
    public WtcClientChannel getChannel(WtcInt32 id)
    {
        synchronized (mChannelsAll)
        {
            return (WtcClientChannel) mChannelsAll.get(id);
        }
    }

    /**
     * @param name name of a channel
     * @return the WtcClientChannel with the given name
     * @throws NoSuchElementException if a channel with the given name does not exist
     */
    public WtcClientChannel getChannel(String name) throws NoSuchElementException
    {
        synchronized (mChannelsAll)
        {
            Enumeration channels = mChannelsAll.elements();
            while (channels.hasMoreElements())
            {
                WtcClientChannel channel = (WtcClientChannel) channels.nextElement();
                if (channel.getName().equalsIgnoreCase(name))
                {
                    return channel;
                }
            }
            throw new NoSuchElementException();
        }
    }

    private WtcpChannelIdList createChannelIdList(WtcInt32 id)
    {
        WtcpChannelIdList list = new WtcpChannelIdList();
        list.addElement(id);
        return list;
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(int channelId, boolean on)
    {
        return activate(WtcInt32.valueOf(channelId, true), on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(int channelId, int opType)
    {
        return activate(WtcInt32.valueOf(channelId, true), opType);
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcInt32 channelId, boolean on)
    {
        return activate(channelId, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcInt32 channelId, int opType)
    {
        return activate(createChannelIdList(channelId), opType);
    }

    /**
     * @param channelIds
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcpChannelIdList channelIds, boolean on)
    {
        return activate(channelIds, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelIds
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer activate(WtcpChannelIdList channelIds, int opType)
    {
        return doChannelOperationRequest(channelIds, WtcpOpCode.ChannelSetActive, opType);
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(int channelId, boolean on)
    {
        return talk(WtcInt32.valueOf(channelId, true), on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(int channelId, int opType)
    {
        return talk(WtcInt32.valueOf(channelId, true), opType);
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(WtcInt32 channelId, boolean on)
    {
        return talk(channelId, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(WtcInt32 channelId, int opType)
    {
        return talk(createChannelIdList(channelId), opType);
    }

    /**
     * @param channelIds
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(WtcpChannelIdList channelIds, boolean on)
    {
        return talk(channelIds, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelIds
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer talk(WtcpChannelIdList channelIds, int opType)
    {
        return doChannelOperationRequest(channelIds, WtcpOpCode.ChannelPushToTalk, opType);
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(int channelId, boolean on)
    {
        return mute(WtcInt32.valueOf(channelId, true), on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(int channelId, int opType)
    {
        return mute(WtcInt32.valueOf(channelId, true), opType);
    }

    /**
     * @param channelId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(WtcInt32 channelId, boolean on)
    {
        return mute(channelId, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelId
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(WtcInt32 channelId, int opType)
    {
        return mute(createChannelIdList(channelId), opType);
    }

    /**
     * @param channelIds
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(WtcpChannelIdList channelIds, boolean on)
    {
        return mute(channelIds, on ? ChannelOperationType.ON : ChannelOperationType.OFF);
    }

    /**
     * @param channelIds
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer mute(WtcpChannelIdList channelIds, int opType)
    {
        return doChannelOperationRequest(channelIds, WtcpOpCode.ChannelMute, opType);
    }

    // TODO:(pv) Make this an operation "builder" concept

    /**
     * @param channelIds the channel ids to set or unset
     * @param opCode WtcpOpCode.ChannelSetActive, WtcpOpCode.ChannelPushToTalk, or WtcpOpCode.ChannelMute
     * @param opType ChannelOperationType.OFF, ChannelOperationType.ON, or ChannelOperationType.ABSOLUTE
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer doChannelOperationRequest(WtcpChannelIdList channelIds, int opCode, int opType)
    {
        try
        {
            WtcLog.debug(TAG, "+doChannelOperationRequest(" + channelIds + ", " + WtcpOpCode.toString(opCode) + ", "
                            + ChannelOperationType.toString(opType) + ")");

            synchronized (mChannelsAll)
            {
                if (channelIds == null)
                {
                    channelIds = new WtcpChannelIdList();
                }

                // We will rebuild this from the current state inside the below while loop
                switch (opCode)
                {
                    case WtcpOpCode.ChannelPushToTalk:
                    {
                        mChannelsPttOningOrOn.removeAllElements();
                        break;
                    }
                }

                WtcClientChannel channel;
                WtcInt32 channelId;

                WtcpChannelIdList channelsRequest = new WtcpChannelIdList();

                Enumeration it = mChannelsAll.elements();
                while (it.hasMoreElements())
                {
                    channel = (WtcClientChannel) it.nextElement();

                    if (VERBOSE_LOG)
                    {
                        WtcLog.info(TAG, "TX BEFORE channel=" + channel);
                    }

                    channelId = channel.getIdSession();

                    switch (opCode)
                    {
                        case WtcpOpCode.ChannelSetActive:
                        {
                            switch (opType)
                            {
                                case ChannelOperationType.ABSOLUTE:
                                {
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mActive.setOning();
                                        channelsRequest.addElement(channelId);
                                    }
                                    else
                                    {
                                        channel.mActive.setOffing();
                                    }
                                    break;
                                }
                                case ChannelOperationType.ON:
                                {
                                    // Mark the channel as activating and add to request list
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mActive.setOning();
                                        channelsRequest.addElement(channelId);
                                    }
                                    else if (channel.mActive.isOningOrOn() && !channel.mActive.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);
                                    }
                                    break;
                                }
                                case ChannelOperationType.OFF:
                                {
                                    if (channelIds.size() == 0 || channelIds.contains(channelId))
                                    {
                                        // Mark the channel as deactivating and *DON'T* add to request list
                                        channel.mActive.setOffing();
                                    }
                                    else if (channel.mActive.isOningOrOn() && !channel.mActive.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);
                                    }
                                    break;
                                }
                            }
                            break;
                        }
                        case WtcpOpCode.ChannelPushToTalk:
                        {
                            switch (opType)
                            {
                                case ChannelOperationType.ABSOLUTE:
                                {
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mPtt.setOning();
                                        channelsRequest.addElement(channelId);
                                    }
                                    else
                                    {
                                        channel.mPtt.setOffing();
                                    }
                                    break;
                                }
                                case ChannelOperationType.ON:
                                {
                                    // Mark the channel as requesting ptt on and add to request list
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mPtt.setOning();
                                        channelsRequest.addElement(channelId);

                                        mChannelsPttOningOrOn.addElement(channelId);
                                    }
                                    else if (channel.mPtt.isOningOrOn() && !channel.mPtt.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);

                                        mChannelsPttOningOrOn.addElement(channelId);
                                    }
                                    break;
                                }
                                case ChannelOperationType.OFF:
                                {
                                    if (channelIds.size() == 0 || channelIds.contains(channelId))
                                    {
                                        // Mark the channel as requesting ptt off and *DON'T* add to request list
                                        channel.mPtt.setOffing();
                                    }
                                    else if (channel.mPtt.isOningOrOn() && !channel.mPtt.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);

                                        mChannelsPttOningOrOn.addElement(channelId);
                                    }
                                    break;
                                }
                            }
                            break;
                        }
                        case WtcpOpCode.ChannelMute:
                        {
                            switch (opType)
                            {
                                case ChannelOperationType.ABSOLUTE:
                                {
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mMute.setOning();
                                        channelsRequest.addElement(channelId);
                                    }
                                    else
                                    {
                                        channel.mMute.setOffing();
                                    }
                                    break;
                                }
                                case ChannelOperationType.ON:
                                {
                                    // Mark the channel as muting and add to request list
                                    if (channelIds.contains(channelId))
                                    {
                                        channel.mMute.setOning();
                                        channelsRequest.addElement(channelId);
                                    }
                                    else if (channel.mMute.isOningOrOn() && !channel.mMute.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);
                                    }
                                    break;
                                }
                                case ChannelOperationType.OFF:
                                {
                                    if (channelIds.size() == 0 || channelIds.contains(channelId))
                                    {
                                        // Mark the channel as unmuting and *DON'T* add to request list
                                        channel.mMute.setOffing();
                                    }
                                    else if (channel.mMute.isOningOrOn() && !channel.mMute.isOffing())
                                    {
                                        channelsRequest.addElement(channelId);
                                    }
                                    break;
                                }
                            }
                            break;
                        }
                    }

                    if (VERBOSE_LOG)
                    {
                        WtcLog.info(TAG, "TX AFTER channel=" + channel);
                    }
                }

                Integer transactionId = null;

                switch (opCode)
                {
                    case WtcpOpCode.ChannelSetActive:
                    {
                        transactionId = mClient.channelsSetActive(channelsRequest);
                        break;
                    }
                    case WtcpOpCode.ChannelPushToTalk:
                    {
                        transactionId = mClient.channelsPushToTalk(channelsRequest);
                        break;
                    }
                    case WtcpOpCode.ChannelMute:
                    {
                        transactionId = mClient.channelsMute(channelsRequest);
                        break;
                    }
                }

                if (VERBOSE_LOG)
                {
                    String temp = null;
                    if (transactionId != null)
                    {
                        temp = "0x" + WtcString.toHexString(transactionId.intValue(), 2) + '(' + transactionId.intValue() + ')';
                    }
                    WtcLog.info(TAG, "TX transactionId=" + temp);
                }

                // TODO:(pv) Save transaction ID in to opCode map
                return transactionId;
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-doChannelOperationRequest(" + channelIds + ", " + WtcpOpCode.toString(opCode) + ", "
                            + ChannelOperationType.toString(opType) + ")");
        }
    }

    /**
     * Called by WtcClient onChannelSetActive, onChannelPushToTalk, and onChannelMute
     * @param client
     * @param listener
     * @param controlHeader
     * @param channelIdErrors
     * @return Map&lt;opCode,transactionId&lt; of any resulting auto-operations
     */
    protected WtcIntegerObjectMapPlatform onChannelOperation(WtcClient client, WtcClientListener listener, //
                    WtcpControlHeader controlHeader, WtcpChannelIdErrorDictionary channelIdErrors)
    {
        try
        {
            WtcLog.debug(TAG, "+onChannelOperation: controlHeader=" + controlHeader + ", channelIdErrors=" + channelIdErrors);

            WtcIntegerObjectMapPlatform allAutoOpChannelIds = new WtcIntegerObjectMapPlatform();

            synchronized (mChannelsAll)
            {
                int transactionId = controlHeader.transactionId;
                int opCode = controlHeader.getOpCode();
                int opType = controlHeader.getOpType();
                boolean isUnsolicited = controlHeader.isUnsolicited();

                if (VERBOSE_LOG)
                {
                    String temp = "0x" + WtcString.toHexString(transactionId, 2) + '(' + transactionId + ')';
                    WtcLog.info(TAG, "RX transactionId=" + temp);
                }

                if (isUnsolicited)
                {
                    WtcLog.info(TAG, "RX UNSOLICITED " + WtcpOpCode.toString(opCode));
                }

                // TODO:(pv) Filter out all but the latest opcode transaction id?

                WtcClientChannel channel;
                WtcInt32 channelId;
                WtcpErrorCode errorCode;
                boolean hasErrorCode;
                boolean autoMuteChannels = AUTO_MUTE_CHANNELS;
                boolean alreadyOn;
                boolean isLatestChannelOperationOningOrOn;

                Enumeration it = mChannelsAll.elements();
                while (it.hasMoreElements())
                {
                    channel = (WtcClientChannel) it.nextElement();
                    channelId = channel.getIdSession();

                    if (VERBOSE_LOG)
                    {
                        WtcLog.info(TAG, "RX BEFORE channel=" + channel);
                    }

                    errorCode = (WtcpErrorCode) channelIdErrors.get(channelId);
                    hasErrorCode = (errorCode != null);
                    if (hasErrorCode)
                    {
                        if (errorCode.isOK())
                        {
                            //
                            // NOTE: channel was already added to channelsTalking when request was made (optimistic)
                            //

                            switch (opCode)
                            {
                                case WtcpOpCode.ChannelSetActive:
                                {
                                    if (autoMuteChannels)
                                    {
                                        WtcpChannelIdList autoOpChannelIds;
                                        autoOpChannelIds = (WtcpChannelIdList) allAutoOpChannelIds.get(WtcpOpCode.ChannelMute);
                                        if (autoOpChannelIds == null)
                                        {
                                            autoOpChannelIds = new WtcpChannelIdList();
                                        }
                                        autoOpChannelIds.addElement(channelId);
                                        allAutoOpChannelIds.put(WtcpOpCode.ChannelMute, autoOpChannelIds);
                                    }

                                    alreadyOn = channel.mActive.setOn(true);
                                    if (!alreadyOn)
                                    {
                                        if (VERBOSE_LOG)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId
                                                            + " is in response, succeeded, and NOT already on; firing event");
                                        }
                                        if (listener != null)
                                        {
                                            listener.onChannelActivated(client, opType, transactionId, //
                                                            channel);
                                        }
                                    }
                                    else
                                    {
                                        if (VERBOSE_LOG)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId
                                                            + " is in response, succeeded, but already on; ignoring");
                                        }
                                    }
                                    break;
                                }
                                case WtcpOpCode.ChannelPushToTalk:
                                {
                                    // TODO:(pv) This "ChannelPushToTalk" logic may need to be duplicated in the ChannelSetActive and ChannelMute cases.
                                    //  The concern is fairly low since those cases aren't thrashed very often, if ever.
                                    //  Also, this code isn't fully vetted yet; no need to duplicate it if it isn't working! :)

                                    // Is the channel OFF or ON in the latest call to doChannelOperation?
                                    isLatestChannelOperationOningOrOn = mChannelsPttOningOrOn.contains(channelId);
                                    if (VERBOSE_LOG)
                                    {
                                        WtcLog.info(TAG, "channel " + channelId + ": isLatestChannelOperationOningOrOn="
                                                        + isLatestChannelOperationOningOrOn);
                                    }

                                    if (isLatestChannelOperationOningOrOn)
                                    {
                                        //
                                        // Channel was ONing or ON
                                        //

                                        alreadyOn = channel.mPtt.setOn(true);
                                        if (!alreadyOn)
                                        {
                                            if (VERBOSE_LOG)
                                            {
                                                WtcLog.warn(TAG, "channel " + channelId
                                                                + " is ONing, in response and succeeded; firing event");
                                            }
                                            if (listener != null)
                                            {
                                                listener.onChannelTalkStarted(client, opType, transactionId, //
                                                                channel);
                                            }
                                        }
                                        else
                                        {
                                            if (VERBOSE_LOG)
                                            {
                                                WtcLog.warn(TAG, "channel " + channelId
                                                                + " is ONing, in response, succeeded, but already ON; ignoring");
                                            }
                                        }
                                    }
                                    else
                                    {
                                        //
                                        // Channel was OFFing or OFF
                                        //

                                        if (channel.mPtt.isOffing())
                                        {
                                            //
                                            // Channel was OFFing; fire an OFF event for a channel that is not yet OFF
                                            //

                                            channel.mPtt.setOn(false);

                                            if (VERBOSE_LOG)
                                            {
                                                WtcLog.warn(TAG, "channel " + channelId
                                                                + " is OFFing, in response and succeeded; firing event");
                                            }
                                            if (listener != null)
                                            {
                                                listener.onChannelTalkStopped(client, opType, transactionId, //
                                                                channel, errorCode);
                                            }
                                        }
                                        else
                                        {
                                            //
                                            // Channel was OFF; don't fire an OFF event for a channel that is already OFF
                                            //

                                            if (VERBOSE_LOG)
                                            {
                                                WtcLog.warn(TAG, "channel " + channelId
                                                                + " is *NOT* OFFing, in response and succeeded; ignoring event");
                                            }
                                        }
                                    }
                                    break;
                                }
                                case WtcpOpCode.ChannelMute:
                                {
                                    alreadyOn = channel.mMute.setOn(true);
                                    if (!alreadyOn)
                                    {
                                        if (VERBOSE_LOG)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId
                                                            + " is in response and succeeded; firing event");
                                        }
                                        if (listener != null)
                                        {
                                            listener.onChannelMuteOn(client, opType, transactionId, //
                                                            channel);
                                        }
                                    }
                                    else
                                    {
                                        if (VERBOSE_LOG)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId
                                                            + " is in response, succeeded, but already on; ignoring");
                                        }
                                    }
                                    break;
                                }
                            }
                        }
                        else
                        {
                            // errorCode != OK

                            if (VERBOSE_LOG)
                            {
                                WtcLog.warn(TAG, "channel " + channelId + " is in response and failed w/ errorCode "
                                                + errorCode + "; firing event");
                            }

                            switch (controlHeader.getOpCode())
                            {
                                case WtcpOpCode.ChannelSetActive:
                                {
                                    mChannelsPttOningOrOn.removeElement(channelId);
                                    channel.mActive.setOn(false);
                                    if (listener != null)
                                    {
                                        listener.onChannelDeactivated(client, opType, transactionId, //
                                                        channel, errorCode);
                                    }
                                    break;
                                }
                                case WtcpOpCode.ChannelPushToTalk:
                                {
                                    mChannelsPttOningOrOn.removeElement(channelId);
                                    channel.mPtt.setOn(false);
                                    if (listener != null)
                                    {
                                        listener.onChannelTalkStopped(client, opType, transactionId, //
                                                        channel, errorCode);
                                    }
                                    break;
                                }
                                case WtcpOpCode.ChannelMute:
                                {
                                    channel.mMute.setOn(false);
                                    if (listener != null)
                                    {
                                        listener.onChannelMuteOff(client, opType, transactionId, //
                                                        channel, errorCode);
                                    }
                                    break;
                                }
                            }
                        }
                    }
                    else
                    {
                        // !hasErrorCode
                        // The RXed channel operation is either not relevant to this channel,
                        // or is Unsolicited [and thus potentially relevant to ALL channels].

                        // NOTE:(pv) "Unsolicited Off" always affects all "OningOrOn" channels

                        switch (opCode)
                        {
                            case WtcpOpCode.ChannelSetActive:
                            {
                                if (channel.mActive.isOningOrOn() && (channel.mActive.isOffing() || isUnsolicited))
                                //if (channel.mActive.isOffing() || isUnsolicited)
                                {
                                    channel.mActive.setOn(false);

                                    mChannelsPttOningOrOn.removeElement(channelId);

                                    if (VERBOSE_LOG)
                                    {
                                        if (isUnsolicited)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn but RXed Unsolicited Off; FIRING EVENT");
                                        }
                                        else
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn and Offing but NOT in response; FIRING EVENT");
                                        }
                                    }

                                    if (listener != null)
                                    {
                                        listener.onChannelDeactivated(client, opType, transactionId, //
                                                        channel, WtcpErrorCode.OK);
                                    }
                                }
                                else
                                {
                                    if (VERBOSE_LOG)
                                    {
                                        WtcLog.warn(TAG, "channel " + channelId //
                                                        + " was NOT in response and *NOT* offing; ignoring");
                                    }
                                }
                                break;
                            }
                            case WtcpOpCode.ChannelPushToTalk:
                            {
                                if (channel.mPtt.isOningOrOn() && (channel.mPtt.isOffing() || isUnsolicited))
                                //if (channel.mPtt.isOffing() || isUnsolicited)
                                {
                                    channel.mPtt.setOn(false);

                                    mChannelsPttOningOrOn.removeElement(channelId);

                                    if (VERBOSE_LOG)
                                    {
                                        if (isUnsolicited)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn but RXed Unsolicited Off; FIRING EVENT");
                                        }
                                        else
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn and Offing but NOT in response; FIRING EVENT");
                                        }
                                    }

                                    if (listener != null)
                                    {
                                        listener.onChannelTalkStopped(client, opType, transactionId, //
                                                        channel, WtcpErrorCode.OK);
                                    }
                                }
                                else
                                {
                                    if (VERBOSE_LOG)
                                    {
                                        WtcLog.warn(TAG, "channel " + channelId //
                                                        + " was NOT in response and *NOT* offing; ignoring");
                                    }
                                }
                                break;
                            }
                            case WtcpOpCode.ChannelMute:
                            {
                                if (channel.mMute.isOningOrOn() && (channel.mMute.isOffing() || isUnsolicited))
                                //if (channel.mMute.isOffing() || isUnsolicited)
                                {
                                    channel.mMute.setOn(false);

                                    if (VERBOSE_LOG)
                                    {
                                        if (isUnsolicited)
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn but RXed Unsolicited Off; FIRING EVENT");
                                        }
                                        else
                                        {
                                            WtcLog.warn(TAG, "channel " + channelId //
                                                            + " was OningOrOn and Offing but NOT in response; FIRING EVENT");
                                        }
                                    }

                                    if (listener != null)
                                    {
                                        listener.onChannelMuteOff(client, opType, transactionId, //
                                                        channel, WtcpErrorCode.OK);
                                    }
                                }
                                else
                                {
                                    if (VERBOSE_LOG)
                                    {
                                        WtcLog.warn(TAG, "channel " + channelId //
                                                        + " was NOT in response and *NOT* offing; ignoring");
                                    }
                                }
                                break;
                            }
                        }
                    }

                    if (VERBOSE_LOG)
                    {
                        WtcLog.info(TAG, "RX AFTER channel=" + channel);
                    }
                }
            }

            WtcIntegerObjectMapPlatform allAutoOpTransactionIds = null;
            if (allAutoOpChannelIds.size() > 0)
            {
                Integer transactionId;

                allAutoOpTransactionIds = new WtcIntegerObjectMapPlatform();

                int opCode;
                WtcpChannelIdList channelIds;

                Enumeration keys = allAutoOpChannelIds.keys();
                while (keys.hasMoreElements())
                {
                    opCode = ((Integer) keys.nextElement()).intValue();
                    channelIds = (WtcpChannelIdList) allAutoOpChannelIds.get(opCode);

                    // TODO:(pv) Support auto-operation "OFF" if needed

                    switch (opCode)
                    {
                        case WtcpOpCode.ChannelMute:
                        {
                            transactionId = doChannelOperationRequest(channelIds, opCode, ChannelOperationType.ON);
                            allAutoOpTransactionIds.put(opCode, transactionId);
                            break;
                        }
                    }
                }
            }
            return allAutoOpTransactionIds;
        }
        finally
        {
            WtcLog.debug(TAG, "-onChannelOperation: controlHeader=" + controlHeader + ", channelIdErrors=" + channelIdErrors);
        }
    }

    public Integer getChannelProperty(WtcInt32 channelId, String key)
    {
        return getChannelProperties(channelId, (WtcString.isNullOrEmpty(key)) ? null : new String[]
        {
            key
        });
    }

    public static final String[] CHANNEL_PROPERTIES_ALL = new String[]
                                                        {
                                                            "*"
                                                        };

    public Integer getChannelProperties(WtcInt32 channelId, String[] keys)
    {
        if (keys == null)
        {
            keys = CHANNEL_PROPERTIES_ALL;
        }

        return getChannelProperties(channelId, new WtcpStringList(keys));
    }

    public Integer getChannelProperties(WtcInt32 channelId, WtcpStringList keys)
    {
        return mClient.channelPropertiesGet(channelId, keys);
    }

    public Integer setChannelProperty(WtcInt32 channelId, String key, String value)
    {
        if (WtcString.isNullOrEmpty(key))
        {
            throw new IllegalArgumentException("key cannot be null or empty");
        }

        if (WtcString.isNullOrEmpty(value))
        {
            value = "";
        }

        WtcpKeyValueList properties = new WtcpKeyValueList();
        properties.put(key, value);
        return setChannelProperties(channelId, properties);
    }

    public Integer setChannelProperties(WtcInt32 channelId, WtcpKeyValueList properties)
    {
        return mClient.channelPropertiesSet(channelId, properties);
    }

    public void onChannelPropertiesGet(WtcClient client, WtcClientListener listener, //
                    WtcpControlHeader controlHeader, int channelId, WtcpKeyValueList keyValues)
    {
        WtcLog.debug(TAG, "onChannelPropertiesGet: controlHeader=" + controlHeader + ", channelId=" + channelId
                        + ", keyValues=" + keyValues);

        WtcClientChannel channel = getChannel(channelId);
        channel.updateProperties(keyValues);

        if (listener != null)
        {
            listener.onChannelPropertiesGet(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            channel, keyValues);
        }
    }

    public void onChannelPropertiesGet(WtcClient client, WtcClientListener listener, //
                    WtcpControlHeader controlHeader, int channelId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onChannelPropertiesGetError: channelId=" + channelId + ", errorCode=" + errorCode);
        WtcClientChannel channel = getChannel(channelId);

        // TODO:(pv) Process channel properties, passing non-well-known up to listener

        if (listener != null)
        {
            listener.onChannelPropertiesGetError(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            channel, errorCode);
        }
    }

    public void onChannelPropertiesSet(WtcClient client, WtcClientListener listener, //
                    WtcpControlHeader controlHeader, int channelId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onChannelPropertiesSet: channelId=" + channelId + ", errorCode=" + errorCode);
        WtcClientChannel channel = getChannel(channelId);

        // TODO:(pv) Process channel properties, passing non-well-known up to listener

        if (listener != null)
        {
            listener.onChannelPropertiesSet(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            channel, errorCode);
        }
    }

    public void onChannelActivity(WtcClient client, WtcClientListener listener, WtcpControlHeader controlHeader,
                    WtcpChannelActivity channelActivity)
    {
        //WtcLog.debug(TAG, "onChannelActivity: controlHeader=" + controlHeader + ", channelActivity=" + channelActivity);

        int channelId = channelActivity.channelId;
        WtcClientChannel channel = getChannel(channelId);
        channel.updateActivity(channelActivity.channelFlags, channelActivity.endpointCount, channelActivity.activityEndpoints);

        if (listener != null)
        {
            // Newer WtcClientListener logic
            if (!listener.onChannelActivity(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            channel, channelActivity))
            {
                // Older WtcClientListener logic
                listener.onChannelActivity(client, controlHeader.getOpType(), controlHeader.transactionId, //
                                channel, channelActivity.channelFlags, channelActivity.activityEndpoints);
            }
        }
    }

    public void onChannelActivity(WtcClient client, WtcClientListener listener, WtcpControlHeader controlHeader,
                    WtcpErrorCode errorCode)
    {
        WtcLog.warn(TAG, "onChannelActivity: controlHeader=" + controlHeader + ", errorCode=" + errorCode);

        // TODO:(pv) Implement this if anyone ever actually uses "Request ChannelActivity"

        if (listener != null)
        {
            listener.onChannelActivityError(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            errorCode);
        }
    }

    public void onChannelChange(WtcClient client, WtcClientListener listener, //
                    WtcpControlHeader controlHeader, boolean reconnect, int change, int channelId)
    {
        WtcInt32 channelKey = WtcInt32.valueOf(channelId, true);

        switch (change)
        {
            case WtcpChannelChange.Added:
            {
                // Temporary until OnChannelPropertiesGet gives us the real name
                String name = "channelId" + channelId;

                WtcpChannelInfo channelInfo = new WtcpChannelInfo(channelId, WtcpChannelFlags.None, name);
                WtcClientChannel channel = new WtcClientChannel(this, channelInfo);
                if (AUTO_MUTE_CHANNELS)
                {
                    channel.mute(true);
                }
                synchronized (mChannelsAll)
                {
                    mChannelsAll.put(channelKey, channel);
                }
                break;
            }
            case WtcpChannelChange.Removed:
            {
                synchronized (mChannelsAll)
                {
                    mChannelsAll.remove(channelKey);
                    mChannelsPttOningOrOn.removeElement(channelKey);
                }
                break;
            }
            case WtcpChannelChange.Changed:
            {
                // ignore
                break;
            }
        }

        if (listener != null)
        {
            listener.onChannelChange(client, controlHeader.getOpType(), controlHeader.transactionId, //
                            reconnect, change, channelId);
        }
    }
}
