package com.twilio.voice;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.util.HashSet;
import java.util.Set;
import tvo.webrtc.audio.AudioDeviceModule;
import tvo.webrtc.audio.JavaAudioDeviceModule;

class MediaFactory {
    private static final String RELEASE_MESSAGE_TEMPLATE = "MediaFactory released %s unavailable";
    private static final Logger logger = Logger.getLogger(MediaFactory.class);
    private static volatile MediaFactory instance;
    private static volatile Set<Object> mediaFactoryOwners = new HashSet<>();

    private long nativeMediaFactoryHandle;
    private AudioDeviceModule audioDeviceModule;
    private AudioDeviceProxy audioDeviceProxy;

    static MediaFactory instance(@NonNull Object owner, @NonNull Context context) {
        Preconditions.checkNotNull(owner, "owner must not be null");
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkApplicationContext(context);
        long nativeMediaFactoryHandle = 0;
        synchronized (MediaFactory.class) {
            if (instance == null) {
                Voice.loadLibrary(context);
                AudioDevice audioDevice = Voice.getAudioDevice();
                if (audioDevice instanceof DefaultAudioDevice) {
                    AudioDeviceModule audioDeviceModule =
                            JavaAudioDeviceModule.builder(context)
                                    .setUseHardwareNoiseSuppressor(
                                            ((DefaultAudioDevice) audioDevice)
                                                    .useHardwareNoiseSuppressor())
                                    .setUseHardwareAcousticEchoCanceler(
                                            ((DefaultAudioDevice) audioDevice)
                                                    .useHardwareAcousticEchoCanceler())
                                    .createAudioDeviceModule();
                    long nativeAudioDeviceModule =
                            audioDeviceModule.getNativeAudioDeviceModulePointer();
                    nativeMediaFactoryHandle = nativeCreate(context, nativeAudioDeviceModule);
                    if (nativeMediaFactoryHandle == 0) {
                        logger.e("Failed to instance MediaFactory");
                    } else {
                        instance = new MediaFactory(audioDeviceModule, nativeMediaFactoryHandle);
                    }
                } else {
                    AudioFormat capturerAudioFormat = audioDevice.getCapturerFormat();
                    AudioFormat renderAudioFormat = audioDevice.getRendererFormat();
                    instance = new MediaFactory();
                    nativeMediaFactoryHandle =
                            nativeCreateWithCustomDevice(
                                    instance,
                                    context,
                                    audioDevice,
                                    capturerAudioFormat,
                                    renderAudioFormat);
                    instance.nativeMediaFactoryHandle = nativeMediaFactoryHandle;
                }
            }
            mediaFactoryOwners.add(owner);
        }

        return instance;
    }

    synchronized @Nullable LocalAudioTrack createAudioTrack(
            Context context, boolean enabled, @NonNull AudioOptions audioOptions, String name) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(audioOptions, "audioOptions must not be null");
        Preconditions.checkApplicationContext(
                context, "must create local audio track with application context");
        Preconditions.checkState(
                nativeMediaFactoryHandle != 0, RELEASE_MESSAGE_TEMPLATE, "createAudioTrack");
        return nativeCreateAudioTrack(
                nativeMediaFactoryHandle, context, enabled, audioOptions, name);
    }

    void release(Object owner) {
        if (instance != null) {
            synchronized (MediaFactory.class) {
                mediaFactoryOwners.remove(owner);
                if (instance != null && mediaFactoryOwners.isEmpty()) {
                    // Release the native audio device module
                    if (audioDeviceModule != null) {
                        audioDeviceModule.release();
                    }
                    if (audioDeviceProxy != null) {
                        audioDeviceProxy.release();
                    }
                    audioDeviceProxy = null;
                    // Release native media factory
                    nativeRelease(nativeMediaFactoryHandle);
                    nativeMediaFactoryHandle = 0;
                    instance = null;
                }
            }
        }
    }

    long getNativeMediaFactoryHandle() {
        return nativeMediaFactoryHandle;
    }

    // Set from the native layer
    void setAudioDeviceProxy(AudioDeviceProxy audioDeviceProxy) {
        this.audioDeviceProxy = audioDeviceProxy;
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    static boolean isReleased() {
        synchronized (MediaFactory.class) {
            return instance == null;
        }
    }

    /*
     * Manually release the media factory instance if a test did not properly release it.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    static void manualRelease() {
        synchronized (MediaFactory.class) {
            if (instance != null) {
                mediaFactoryOwners.clear();
                // Provide a dummy object to initiate release process
                Object o = new Object();
                instance.release(o);
            }
        }
    }

    public MediaFactory() {}

    private MediaFactory(AudioDeviceModule audioDeviceModule, long nativeMediaFactoryHandle) {
        this.nativeMediaFactoryHandle = nativeMediaFactoryHandle;
        this.audioDeviceModule = audioDeviceModule;
    }

    private static native long nativeCreate(Context context, long nativeMediaFactoryHandle);

    private static native long nativeCreateWithCustomDevice(
            MediaFactory mediaFactory,
            Context context,
            AudioDevice audioDevice,
            AudioFormat capturerFormat,
            AudioFormat renderFormat);

    private native LocalAudioTrack nativeCreateAudioTrack(
            long nativeMediaFactoryHandle,
            Context context,
            boolean enabled,
            AudioOptions audioOptions,
            String name);

    private native void nativeRelease(long mediaFactoryHandle);
}
