package com.twilio.voice;

import android.content.Context;
import android.os.Handler;
import androidx.annotation.VisibleForTesting;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONObject;

abstract class InternalCall implements EventPublisher.EventPublisherListener {

    private static final Logger logger = Logger.getLogger(InternalCall.class);
    private static final String TEMP_CALL_SID_PREFIX = "TSID";
    String sid;
    String bridgeToken;
    String from;
    String to;
    String selectedRegion = Voice.region;
    String gateway;
    String region;
    String codecParams;
    String selectedCodec;
    Handler handler;
    Context context;
    boolean isMuted;
    boolean isOnHold;
    Call.State state;
    boolean disconnectCalled;
    EventPublisher publisher;
    Constants.Direction direction;
    private JSONArray payload;
    private final UUID uuid = UUID.randomUUID();
    // Generate a temporary call SID for cases where one has not yet been assigned.
    final String tempCallSid = TEMP_CALL_SID_PREFIX + uuid;

    @SuppressWarnings("unused")
    void setSid(String sid) {
        this.sid = sid;
    }

    boolean isValidState() {
        return state != Call.State.DISCONNECTED;
    }

    public abstract String getSid();

    public abstract void sendDigits(String digits);

    public abstract Call.State getState();

    public abstract boolean isMuted();

    public abstract void mute(boolean mute);

    public abstract void disconnect();

    @Override
    public void onError(VoiceException voiceException) {
        logger.e(
                "Error publishing data : "
                        + voiceException.getMessage()
                        + ":"
                        + voiceException.getErrorCode());
    }

    /** This method gets called when a new stats becomes available. */
    void onSample(RTCStatsSample currentSample) {
        payload = InsightsUtils.publishMetrics(currentSample, sid, direction, payload, publisher);
    }

    void publishFeedbackEvent(Call.Score score, Call.Issue issue) {
        logger.d("Publishing event feedback event");
        EventPayload eventPayload;
        String eventName;
        if (score == Call.Score.NOT_REPORTED && issue == Call.Issue.NOT_REPORTED) {
            eventPayload = createEventPayloadBuilder().build();
            eventName = EventType.FEEDBACK_RECEIVED_NONE_EVENT;
        } else {
            eventPayload = createEventPayloadBuilder().score(score).issue(issue).build();
            eventName = EventType.FEEDBACK_RECEIVED_EVENT;
        }

        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            Event event =
                    publisher.createEvent(
                            com.twilio.voice.Constants.SeverityLevel.INFO,
                            EventGroupType.FEEDBACK_EVENT_GROUP,
                            eventName,
                            connectionEventPayload);
            publisher.publish(
                    com.twilio.voice.Constants.SeverityLevel.INFO,
                    EventGroupType.FEEDBACK_EVENT_GROUP,
                    eventName,
                    event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    EventPayload.Builder createEventPayloadBuilder() {
        return new EventPayload.Builder()
                .callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.getClientSdkProductName())
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE);
    }

    EventPayload.Builder createEventPayloadBuilderForSettingsEvent() {
        return createEventPayloadBuilder().codecParams(codecParams).selectedCodec(selectedCodec);
    }

    @VisibleForTesting
    EventPublisher getPublisher() {
        return this.publisher;
    }
}
