package com.twilio.voice;

import android.os.Handler;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.twilio.voice.Call.CallMessageListener;

public class CallMessageListenerProxy implements Call.CallMessageListener {
    public CallMessageListenerProxy(
            @NonNull EventPublisher publisher, @Nullable CallMessageListener callMessageListener) {
        this.publisher = publisher;
        this.callMessageListener = callMessageListener;
    }

    private final CallMessageListener callMessageListener;
    private final EventPublisher publisher;
    private Handler handler = Utils.createHandler();

    @Override
    public void onMessageReceived(final String callSid, final CallMessage callMessage) {
        InsightsUtils.publishCallMessageReceivedEvent(
                publisher,
                new EventPayload.Builder()
                        .productName(Constants.getClientSdkProductName())
                        .callSid(callSid)
                        .payLoadType(Constants.APP_JSON_PAYLOAD_TYPE)
                        .callMessageEventType(Constants.CallMessageEventType.RECEIVED)
                        .voiceEventSid(callMessage.getVoiceEventSID())
                        .build());
        if (null != this.callMessageListener) {
            CallMessageListenerProxy.this.handler.post(
                    () ->
                            CallMessageListenerProxy.this.callMessageListener.onMessageReceived(
                                    callSid, callMessage));
        }
    }

    @Override
    public void onMessageSent(final String callSid, final String voiceEventSID) {
        InsightsUtils.publishCallMessageSentEvent(
                publisher,
                new EventPayload.Builder()
                        .productName(Constants.getClientSdkProductName())
                        .callSid(callSid)
                        .payLoadType(Constants.APP_JSON_PAYLOAD_TYPE)
                        .callMessageEventType(Constants.CallMessageEventType.SENT)
                        .voiceEventSid(voiceEventSID)
                        .build());
        if (null != this.callMessageListener) {
            CallMessageListenerProxy.this.handler.post(
                    () ->
                            CallMessageListenerProxy.this.callMessageListener.onMessageSent(
                                    callSid, voiceEventSID));
        }
    }

    @Override
    public void onMessageFailure(
            final String callSid, final String voiceEventSID, final VoiceException error) {
        InsightsUtils.publishCallMessageSendFailedEvent(
                publisher,
                new EventPayload.Builder()
                        .productName(Constants.getClientSdkProductName())
                        .callSid(callSid)
                        .payLoadType(Constants.APP_JSON_PAYLOAD_TYPE)
                        .voiceEventSid(voiceEventSID)
                        .errorCode((long) error.getErrorCode())
                        .errorMessage(error.getLocalizedMessage())
                        .build());
        if (null != this.callMessageListener) {
            CallMessageListenerProxy.this.handler.post(
                    () ->
                            CallMessageListenerProxy.this.callMessageListener.onMessageFailure(
                                    callSid, voiceEventSID, error));
        }
    }
}
