package com.twilio.video;

import android.os.Handler;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.nio.ByteBuffer;

/**
 * A remote data track represents a unidirectional remote data source from which messages can be
 * received from a participant.
 */
public class RemoteDataTrack extends DataTrack {
    private static final Logger logger = Logger.getLogger(RemoteDataTrack.class);

    private final String sid;
    private long nativeRemoteDataTrackContext;
    private Handler handler;

    /*
     * All native participant callbacks are passed through the listener proxy and atomically
     * forward events to the developer listener.
     */
    private Listener listener;

    /*
     * This listener proxy is bound at the JNI level.
     */
    @SuppressWarnings("unused")
    private final Listener dataTrackListenerProxy =
            new Listener() {

                /*
                 * The onMessage callback is synchronized on the RemoteDataTrack instance in the
                 * notifier and developer thread to ensure the handler and listener are atomically accessed
                 * before notifying the developer.
                 */

                @Override
                public void onMessage(
                        @NonNull final RemoteDataTrack remoteDataTrack,
                        @NonNull final ByteBuffer messageBuffer) {
                    checkCallback(messageBuffer, "onMessage(ByteBuffer)");

                    synchronized (RemoteDataTrack.this) {
                        if (handler != null) {
                            handler.post(
                                    () -> {
                                        synchronized (RemoteDataTrack.this) {
                                            logger.d("onMessage(ByteBuffer)");

                                            if (listener != null) {
                                                listener.onMessage(remoteDataTrack, messageBuffer);
                                            }
                                        }
                                    });
                        }
                    }
                }

                @Override
                public void onMessage(
                        @NonNull final RemoteDataTrack remoteDataTrack,
                        @NonNull final String message) {
                    checkCallback(message, "onMessage(String)");

                    synchronized (RemoteDataTrack.this) {
                        if (handler != null) {
                            handler.post(
                                    () -> {
                                        synchronized (RemoteDataTrack.this) {
                                            logger.d("onMessage(String)");

                                            if (listener != null) {
                                                listener.onMessage(remoteDataTrack, message);
                                            }
                                        }
                                    });
                        }
                    }
                }

                private void checkCallback(Object message, String callback) {
                    Preconditions.checkNotNull(message, "Received null message in %s", callback);
                }
            };

    /**
     * Returns the remote data track's server identifier. This value uniquely identifies the remote
     * data track within the scope of a {@link Room}.
     */
    @NonNull
    public String getSid() {
        return sid;
    }

    /**
     * Set the remote data track listener. The thread on which this method is called is the same
     * thread used to notify of received data track messages.
     *
     * @param listener data track listener
     */
    public synchronized void setListener(@Nullable Listener listener) {
        this.handler = listener != null ? Util.createCallbackHandler() : null;
        this.listener = listener;
    }

    RemoteDataTrack(
            boolean enabled,
            boolean ordered,
            boolean reliable,
            int maxPacketLifeTime,
            int maxRetransmits,
            @NonNull String sid,
            @NonNull String name,
            long nativeRemoteDataTrackContext) {
        super(enabled, ordered, reliable, maxPacketLifeTime, maxRetransmits, name);
        this.sid = sid;
        this.nativeRemoteDataTrackContext = nativeRemoteDataTrackContext;
    }

    synchronized void release() {
        if (!isReleased()) {
            nativeRelease(nativeRemoteDataTrackContext);
            nativeRemoteDataTrackContext = 0;
        }
    }

    boolean isReleased() {
        return nativeRemoteDataTrackContext == 0;
    }

    /** Interface that provides {@link RemoteDataTrack} events. */
    public interface Listener {
        /**
         * This method notifies the listener that a binary message was received.
         *
         * @param remoteDataTrack Remote data track.
         * @param messageBuffer The binary message received.
         */
        void onMessage(@NonNull RemoteDataTrack remoteDataTrack, @NonNull ByteBuffer messageBuffer);

        /**
         * This method notifies the listener that a string message was received.
         *
         * @param remoteDataTrack Remote data track.
         * @param message The string message received.
         */
        void onMessage(@NonNull RemoteDataTrack remoteDataTrack, @NonNull String message);
    }

    private native void nativeRelease(long nativeRemoteDataTrackContext);
}
