package com.twilio.video;

import androidx.annotation.NonNull;
import java.util.HashSet;
import java.util.Set;

/**
 * IceOptions specifies custom media connectivity configurations.
 *
 * <p>Media connections are established using the ICE (Interactive Connectivity Establishment)
 * protocol. These options allow you to customize how data flows to and from participants, and which
 * protocols to use. You may also provide your own ICE servers, overriding the defaults.
 * https://www.twilio.com/stun-turn.
 */
public class IceOptions {
    private final Set<IceServer> iceServers;
    private final IceTransportPolicy iceTransportPolicy;

    private IceOptions(Builder builder) {
        this.iceServers = builder.iceServers;
        this.iceTransportPolicy = builder.iceTransportPolicy;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        IceOptions that = (IceOptions) o;
        return iceServers.equals(that.iceServers) && iceTransportPolicy == that.iceTransportPolicy;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = 31 * result + (iceServers != null ? iceServers.hashCode() : 0);
        result = 31 * result + (iceTransportPolicy != null ? iceTransportPolicy.hashCode() : 0);

        return result;
    }

    @NonNull
    public Set<IceServer> getIceServers() {
        return iceServers;
    }

    @NonNull
    public IceTransportPolicy getIceTransportPolicy() {
        return iceTransportPolicy;
    }

    IceServer[] getIceServersArray() {
        IceServer[] iceServersArray = new IceServer[0];
        if (iceServers != null && iceServers.size() > 0) {
            iceServersArray = iceServers.toArray(new IceServer[iceServers.size()]);
        }
        return iceServersArray;
    }

    public static class Builder {
        private Set<IceServer> iceServers = new HashSet<>();
        private IceTransportPolicy iceTransportPolicy = IceTransportPolicy.ALL;

        public Builder() {}

        /** Set of {@link IceServer} objects to be used during connection establishment. */
        @NonNull
        public Builder iceServers(@NonNull Set<IceServer> iceServers) {
            this.iceServers = iceServers;
            return this;
        }

        /** The transport policy to use. Defaults to {@link IceTransportPolicy#ALL}. */
        @NonNull
        public Builder iceTransportPolicy(@NonNull IceTransportPolicy iceTransportPolicy) {
            this.iceTransportPolicy = iceTransportPolicy;
            return this;
        }

        @NonNull
        public IceOptions build() {
            return new IceOptions(this);
        }
    }
}
